# coding: utf-8
from copy import deepcopy

from flask import (
    g,
    request,
)

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error_not_found,
    json_error_forbidden,
    build_list_response,
)
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    requires,
    no_permission_required,
    scopes_required,
)
from intranet.yandex_directory.src.yandex_directory.core.events import event
from intranet.yandex_directory.src.yandex_directory.core.models.webhook import WebHookModel
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
    uses_out_schema,
)

WEBHOOK_OUT_SCHEMA = {
    'title': 'Web hook',
    'type': 'object',
    'properties': {
        'id': schemas.INTEGER,
        'url': schemas.STRING,
        'service_id': schemas.INTEGER,
        'tvm_client_id': schemas.INTEGER,
        'event_names': {
            'type': ['array'],
            'items': {
                'type': 'string',
                'enum': list(map(str, vars(event)))
            },
            'uniqueItems': True,
        },
        'fields_filter':  {
            'type': 'object',
        },
        'expand_content': schemas.BOOLEAN
    },
    'required': [
        'id',
        'url',
        'service_id',
        'event_names',
        'fields_filter',
        'expand_content',
        'tvm_client_id',
    ],
}

WEBHOOK_CREATE_SCHEMA = deepcopy(WEBHOOK_OUT_SCHEMA)
WEBHOOK_CREATE_SCHEMA['title'] = 'Create webhook'
WEBHOOK_CREATE_SCHEMA['required'] = [
    'url',
]
del WEBHOOK_CREATE_SCHEMA['properties']['id']
del WEBHOOK_CREATE_SCHEMA['properties']['service_id']


class WebHookListView(View):

    @internal
    @uses_schema(WEBHOOK_CREATE_SCHEMA)
    @uses_out_schema(WEBHOOK_OUT_SCHEMA)
    @scopes_required([scope.write_webhooks])
    @no_permission_required
    @requires(org_id=False, user=False)
    def post(self, meta_connection, main_connection, data):
        """
        Добавить подписку на события.

        **Подписка может начать действовать с некоторой задержкой.** (до 300 секунд)
        ---
        tags:
          - Подписки
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Подписка добавлена.
        """
        service_id = g.service.id
        if not service_id:
            return json_error_forbidden()

        url = data['url']
        event_names = data.get('event_names', [])

        existed_webhook = self._get_existed_webhook(
            meta_connection, url, service_id,
        )

        if existed_webhook:
            existed_events = set(existed_webhook['event_names'])
            new_events = set(event_names)
            if existed_events == new_events or new_events.issubset(existed_events):
                webhook = existed_webhook
            else:
                webhook = WebHookModel(meta_connection).update(
                    update_data={'event_names': list(new_events.union(existed_events))},
                    filter_data={'id': existed_webhook['id']}
                )
        else:
            # TODO: валидировать url
            webhook = WebHookModel(meta_connection).create(
                url=data['url'],
                service_id=service_id,
                fields_filter=data.get('fields_filter', {}),
                event_names=event_names,
                expand_content=data.get('expand_content', False),
                tvm_client_id=data.get('tvm_client_id'),
            )

        return json_response(
            data=webhook,
            status_code=201,
        )

    def _get_existed_webhook(self, meta_connection, url, service_id):
        return WebHookModel(meta_connection).filter(
            url=url,
            service_id=service_id,
            environment=app.config['ENVIRONMENT'],
        ).fields('*').one()

    @internal
    @scopes_required([scope.read_webhooks, scope.write_webhooks])
    @no_permission_required
    @requires(org_id=False, user=False)
    def get(self, meta_connection, main_connection):
        """
        Получить список подписок на события.

        Пример выдачи:

            {
                ...
                "result": [
                    ...
                    {
                        'id': 1,
                        'service_id': 1,
                        'event_names': ['department_added'],
                        'expand_content': False,
                        'fields_filter': {
                            'org_id': 1
                        },
                        'url': 'http://someurl.com'
                    }
                    ...
                ]
            }

        ---
        tags:
          - Подписки
        parameters:
          - in: query
            name: page
            type: integer
            description: текущая страница
          - in: query
            name: per_page
            type: integer
            description: какое кол-во объектов выведено на странице
        responses:
          200:
            description: Список подписок.
        """

        response = build_list_response(
            model=WebHookModel(meta_connection),
            path=request.path,
            query_params=request.args.to_dict(),
            model_filters=self._get_filters(),
            max_per_page=1000,
        )

        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def _get_filters(self):
        return {
            'service_id': g.service.id
        }


class WebHookDetailView(View):
    @internal
    @scopes_required([scope.write_webhooks])
    @no_permission_required
    @requires(org_id=False, user=False)
    def delete(self, meta_connection, main_connection, webhook_id):
        """
        Удалить подписку на событие.

        ---
        tags:
          - Подписки
        parameters:
          - in: path
            name: id
            required: true
            type: integer
        responses:
          204:
            description: Подписка удалена.
          404:
            description: Подписка не найдена.
        """
        webhook_filter = {
            'id': webhook_id,
            'service_id': g.service.id
        }
        webhook_model = WebHookModel(meta_connection)
        webhook = webhook_model.find(webhook_filter, one=True)
        if webhook:
            webhook_model.delete(webhook_filter)
            return json_response(
                data={},
                status_code=204
            )
        return json_error_not_found()
