# coding: utf-8

from flask import (
    request,
)

from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError, NotFoundError, InvalidValue
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory import app

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_permission_required,
    internal,
    no_scopes,
    requires,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    GroupModel,
    DepartmentModel,
    DomainModel,
    UserMetaModel,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error_required_field,
    json_error_conflict_fields,
    get_user_data_from_blackbox_by_login,
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shard,
    get_main_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.views.users import create_user_from_bb_info


class DomainOwnedInMultipleOrganizationsError(APIError):
    code = 'org_id_conflict'
    status_code = 409
    message = 'Domain owned by more than one organization: {org_ids}'
    description = 'Домен подтверждён более чем в одной организации: {org_ids}'

    def __init__(self, org_ids):
        super(DomainOwnedInMultipleOrganizationsError, self).__init__(org_ids=org_ids)


class WhoIsView(View):
    methods = ['get']

    def _by_email(self, meta_connection, email):
        name = email.split('@', 1)

        # проверяем, что обе части email непустые строки
        if not name or len(name) != 2 or not name[0] or not name[1]:
            raise InvalidValue(params='email')

        data = []
        user_data = get_user_data_from_blackbox_by_login(email, attributes=['1017'])

        if user_data and not user_data['is_maillist']:
            uid = int(user_data['uid'])
            data = []
            org_ids = UserMetaModel(meta_connection).filter(id=uid, is_dismissed=False).scalar('org_id')
            for org_id in org_ids:
                data.append(
                    {'org_id': org_id, 'type': 'user', 'object_id': uid}
                )

        if data:
            return data

        name, domain_part = name

        # Игнорируем переданный в заголовках org_id, т.к. он совсем не обязательный
        # А если не задан org_id, то пытаемся по доменной части определить номер организации, получить шард,
        # на котором она живет и установить main_connection для запросов к main базе
        domains = DomainModel(None).find(
            filter_data={'name': domain_part, 'owned': True},
            fields=['org_id'],
            one=True
        )

        if not domains:
            raise NotFoundError

        org_id = domains['org_id']
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard) as main_connection:
            department = DepartmentModel(main_connection).find(
                filter_data={'org_id': org_id, 'label': name},
                fields=['id'],
                one=True
            )
            if department:
                object_type = 'department'
                object_id = department['id']
                data = {'org_id': org_id, 'type': object_type, 'object_id': object_id}
                return [data]

            group = GroupModel(main_connection).find(
                filter_data={'org_id': org_id, 'label': name},
                fields=['id'],
                one=True
            )
            if group:
                object_type = 'group'
                object_id = group['id']
                data = {'org_id': org_id, 'type': object_type, 'object_id': object_id}
                return [data]

            # если не нашли ящик в базе добавим создадим его в организации из данных blackbox
            if user_data and not user_data['is_maillist']:
                create_user_from_bb_info(user_data, org_id)
                data = {'org_id': org_id, 'type': 'user', 'object_id': int(user_data['uid'])}
                return [data]

        raise NotFoundError

    def _by_domain(self, domain):
        domains = DomainModel(None).find_all(
            filter_data={'name': domain, 'owned': True},
            fields=['org_id'],  # 'object_id'], #'type'], # 'object_id'],
            one=True
        )
        if len(domains) > 1:
            org_id_set = set([i['org_id'] for i in domains])
            if len(org_id_set) > 1:
                raise DomainOwnedInMultipleOrganizationsError(org_ids=list(org_id_set))

        if len(domains) == 0:
            org_id, object_id = app.domenator.who_is(domain)
        else:
            org_id = domains[0]['org_id']
            object_id = domain

        return [{
            'org_id': org_id,
            'type': "domain",
            'object_id': object_id,
        }]

    @internal
    @no_permission_required
    @no_scopes
    @requires(org_id=False, user=False)
    def get_4(self, meta_connection, main_connection):
        """
        Информация об одной сущности по domain или email.

        Пример 1:

        /v4/who-is/?email=vasya@pupkin.org

        Пример ответа:

            {
              "org_id": 42,

              "type": "user",
              "object_id": 100500
            }

        Пример 2:

        v4/whois-is/?domain=somedomain.com.

        Пример ответа:
        {
            "org_id": 42,
            "type": "domain",
            "object_id":"somedomain.com"
        }

        Поля ответа:

        * **org_id** - идентификатор организации
        * **type** - тип сущности, может принимать значения user, group, department
        * **object_id** - идентификатор сущности
        ---
        tags:
          - Организация
          - Отдел
          - Группа
          - Пользователь
        parameters:
          - in: query
            name: email
            type: string
          - in: query
            name: domain
            type: string
        responses:
          200:
            description: Success
          403:
            description: Request is not avaliable for this service
          404:
            description: Object is not found by email
          422:
            description: Required email or wrong value
        """

        # Если переданы email и domain вместе отдавать 422
        email = request.args.get('email')
        domain = request.args.get('domain')

        if email is not None and domain is not None:
            return json_error_conflict_fields(fields=['email', 'domain'])
        # сразу обработаем ошибку, когда оба параметра пустые
        # было решено запрашивать только один из возможных параметров
        if email is None and domain is None:
            return json_error_required_field('email')

        if email is not None:
            return json_response(
                self._by_email(meta_connection, email)[0]
            )
        elif domain is not None:
            return json_response(
                self._by_domain(domain)[0]
            )

        raise NotFoundError

    @internal
    @no_permission_required
    @no_scopes
    @requires(org_id=False, user=False)
    def get_11(self, meta_connection, main_connection):
        """
        Информация об одной сущности по domain или email.

        Пример 1:

        /v4/who-is/?email=vasya@pupkin.org

        Пример ответа:

            [{
              "org_id": 42,
              "type": "user",
              "object_id": 100500
            }]

        Пример 2:

        v4/whois-is/?domain=somedomain.com.

        Пример ответа:
        [{
            "org_id": 42,
            "type": "domain",
            "object_id":"somedomain.com"
        }]

        Поля ответа:

        * **org_id** - идентификатор организации
        * **type** - тип сущности, может принимать значения user, group, department
        * **object_id** - идентификатор сущности
        ---
        tags:
          - Организация
          - Отдел
          - Группа
          - Пользователь
        parameters:
          - in: query
            name: email
            type: string
          - in: query
            name: domain
            type: string
        responses:
          200:
            description: Success
          403:
            description: Request is not avaliable for this service
          404:
            description: Object is not found by email
          422:
            description: Required email or wrong value
        """

        # Если переданы email и domain вместе отдавать 422
        email = request.args.get('email')
        domain = request.args.get('domain')

        if email is not None and domain is not None:
            return json_error_conflict_fields(fields=['email', 'domain'])
        # сразу обработаем ошибку, когда оба параметра пустые
        # было решено запрашивать только один из возможных параметров
        if email is None and domain is None:
            return json_error_required_field('email')

        if email is not None:
            return json_response(
                self._by_email(meta_connection, email)
            )
        elif domain is not None:
            return json_response(
                self._by_domain(domain)
            )

        raise NotFoundError
