# -*- coding: utf-8 -*-

"""
Клиент для сборщика почты YARM (Yandex Revers Mail)
https://wiki.yandex-team.ru/Pochta/mx/yarm/
"""
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from functools import (
    partial,
)
from retrying import retry

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    url_join,
    log_service_response,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.core.yarm.exceptions import error_reason_to_exception, YarmError
from intranet.yandex_directory.src.yandex_directory.common.utils import get_suid_from_blackbox_for_uid


# id сервиса почта в паспорте
from intranet.yandex_directory.src.yandex_directory.common import http_client

MAIL_SID = 2


def raise_if_error(json_response):
    if 'error' in json_response:
        error_reason = json_response['error'].get('reason', 'unknown')
        exc = error_reason_to_exception.get(error_reason, YarmError)
        raise exc(error_reason)


def get_suid_or_raise(uid):
    suid = get_suid_from_blackbox_for_uid(uid, MAIL_SID)
    if not suid:
        with log.fields(uid=uid):
            log.error('No suid for user')
        raise RuntimeError('No suid for uid')
    return suid


@retry(wait_exponential_multiplier=500,
       wait_exponential_max=10000,
       stop_max_attempt_number=10,
       retry_on_exception=retry_http_errors('yarm'))
def _make_request(method, handle, **kwargs):
    base_url = app.config['YARM_URL']
    # https://wiki.yandex-team.ru/pochta/mx/yarm/#opisanieparametrov
    kwargs['json'] = 1   # для ответа в формает json
    kwargs['mdb'] = 'pg'  # со слов @grascm можно для всех указывать "pg"
    url = url_join(base_url, handle)

    password = kwargs.pop('password', None)
    data = None
    if password:
        data = {'password': password}

    response = http_client.request(method, url, data=data, params=kwargs)
    log_service_response('yarm', method, url, kwargs, response)
    response.raise_for_status()
    json_response = response.json()
    raise_if_error(json_response)
    return json_response


_get = partial(_make_request, 'get')
_post = partial(_make_request, 'post')


def create(uid, login, password, server, port, imap=True, ssl=True, no_delete_msgs=True, sync_abook=True, mark_archive_read=True, user=None):
    """
    Создание сборщика почты
    https://wiki.yandex-team.ru/Pochta/mx/yarm/#zapros/api/create
    :param uid: id аккаунта для собора писем в паспорте
    :param login: логин с которого будет собираться почта
    :param password: пароль ящика с которого будет собираться почта
    :param server: сервер, с которого нужно собирать почту
    :param port: порт на сервере, к которому нужно подключаться
    :param imap: использовать imap иначе pop3
    :param ssl: нужно используя SSL  (по умолчанию Да)
    :param no_delete_msgs: флаг того, что письма с удаленного сервера удалять не надо  (по умолчанию Да)
    :param sync_abook: флаг того, что для сборщика нужно выполнить процедуру сбора контактов  (по умолчанию Да)
    :param mark_archive_read: флаг того, что письма пришедшие в исходный ящик до того, как был создан сборщик, нужно при сборе пометить прочитанными (по умолчанию Да)
    :param user: логин пользователя, который создает сборщик
    :rtype: dict
    """
    if server in app.config['YANDEX_MAIL_SERVERS']:
        assert login != user, 'login must be not equal to user.'
    suid = get_suid_or_raise(uid)
    kwargs = {
        'suid': suid,
        'login': login,
        'password': password,
        'server': server,
        'port': port,
        'ssl': int(bool(ssl)),  # 0 или 1
        'imap': int(bool(imap)),  # 0 или 1
        'no_delete_msgs': int(bool(no_delete_msgs)),  # 0 или 1
        'sync_abook': int(bool(sync_abook)),  # 0 или 1
        'mark_archive_read': int(bool(mark_archive_read)),  # 0 или 1
    }
    if user:
        kwargs['user'] = user
    return _post('/api/create', **kwargs)


def check_server(login, password, server, port, uid=None, imap=True, ssl=True):
    """
    Проверяет правильность настроек сборщика
    https://wiki.yandex-team.ru/Pochta/mx/yarm/#zapros/api/check_server
    :param uid:  id аккаунта для собора писем в паспорте
    :param login: логин с которого будет собираться почта
    :param password: пароль ящика с которого будет собираться почта
    :param server: сервер, с которого нужно собирать почту
    :param port: порт на сервере, к которому нужно подключаться
    :param ssl: нужно используя SSL (по умолчанию Да)
    :param imap: по умолчанию True. Если False, значит pop3
    :rtype: dict
    """
    kwargs = {
        'login': login,
        'password': password,
        'server': server,
        'port': port,
        'imap': int(bool(imap)),  # 0 или 1
        'ssl': int(bool(ssl)),  # 0 или 1
    }
    if uid:
        kwargs['suid'] = get_suid_or_raise(uid)
    return _post('/api/check_server', **kwargs)


def delete(uid, popid):
    """
    Удаление сборщика
    https://wiki.yandex-team.ru/Pochta/mx/yarm/#zapros/api/delete
    :param uid: id аккаунта для собора писем в паспорте
    :param popid:
    :rtype: dict
    """
    suid = get_suid_or_raise(uid)
    return _post('/api/delete', suid=suid, popid=popid)


def list_all(uid, popid=None):
    """
    Получение списка сборщиков пользователя
    https://wiki.yandex-team.ru/Pochta/mx/yarm/#zapros/api/list
    :param uid: id аккаунта для собора писем в паспорте
    :param popid: идентификатор сборщика (если указан, возвращает информацию по конкретному сборщику)
    :rtype: dict
    """
    suid = get_suid_or_raise(uid)
    kwargs = {}
    if popid:
        kwargs['popid'] = popid
    return _get('/api/list', suid=suid, **kwargs)


def status(popid):
    """
    Статистика по одному сборщику
    https://wiki.yandex-team.ru/Pochta/mx/yarm/#zapros/api/status
    :param popid: идентификатор сборщика
    :rtype: dict
    """
    return _get('/api/status', popid=popid)


