from contextlib import contextmanager
from urllib.parse import urlparse

import re
import time

from intranet.yandex_directory.src.yandex_directory.core.utils.credential_sanitizer import CredentialSanitizer
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import requests_log, default_log
from intranet.yandex_directory.src.yandex_directory import app


class HttpRequestLogRecord(object):
    def __init__(self, method, url, headers):
        self.start_time = time.time()
        self.method = method
        self.url = url
        self.host = urlparse(self.url).netloc
        self.headers = headers

    def finish(self, response=None):
        request_time = time.time() - self.start_time
        response_code = None

        if response is not None:
            if hasattr(response, 'status_code'):
                response_code = response.status_code

            if response_code is None:
                default_log.warn('The resposne status could not be determined, host is %s' % self.host)

        requests_log.debug(
            '%s %s\nHEADERS:\n%s\nRESPONSE CODE: %s' % (
                self.method,
                CredentialSanitizer.sanitize_all_parameter_in_url(self.url),
                CredentialSanitizer.get_headers_list_with_sanitized_credentials(self.headers),
                response_code,
            ),
            extra={
                'request_time': request_time,
            },
        )

        prepare_host = re.sub('\\W', '_', self.host)
        app.stats_aggregator.add_to_bucket(
            metric_name='service_%s_response_time' % prepare_host,
            value=request_time,
        )
        app.stats_aggregator.inc('service_%s_%s_response_summ' % (prepare_host, response_code or 'none'))
        if isinstance(response_code, int) and response_code >= 500:
            app.stats_aggregator.inc('service_%s_response_error_summ' % prepare_host)

    def error(self, exception):
        request_time = time.time() - self.start_time
        requests_log.debug(
            '%s %s\nHEADERS:\n%s\nERROR: %s' % (
                self.method,
                CredentialSanitizer.sanitize_all_parameter_in_url(self.url),
                CredentialSanitizer.get_headers_list_with_sanitized_credentials(self.headers),
                exception.__str__(),
            ),
            extra={
                'request_time': request_time,
            },
        )
        prepare_host = re.sub('\\W', '_', self.host)
        app.stats_aggregator.inc('service_%s_response_error_summ' % prepare_host)


class HttpRequestLogService(object):
    @staticmethod
    def start(method, url, headers):
        return HttpRequestLogRecord(method, url, headers)


@contextmanager
def grpc_logging(host, method):
    log_record = HttpRequestLogService.start(method, 'https://' + host, headers={})
    try:
        yield
        log_record.finish()
    except Exception as e:
        log_record.error(e)
        raise
