# -*- coding: utf-8 -*-


__all__ = [
    'log',
    'SensitiveLogger',
]

import os
import logging
import threading
from contextlib import contextmanager
from collections.abc import Iterable

from copy import copy
from ylog.context import log_context


def make_json_serializable(fields):
    try:
        if fields is None:
            return None
        elif isinstance(fields, int):
            return fields
        elif isinstance(fields, bytes):
            return fields.decode('utf-8')
        elif isinstance(fields, str):
            return fields
        elif isinstance(fields, (list, tuple)):
            return list(map(make_json_serializable, fields))
        elif isinstance(fields, dict):
            fields = copy(fields)
            for field_name, field in fields.items():
                fields[field_name] = make_json_serializable(field)
            return fields
        elif not isinstance(fields, Iterable):
            return str(fields)
    except Exception as exc:
        log.exception('Unserializable: {}'.format(str(exc)))
    return 'unknown unserializable type "{}"'.format(type(fields))


class TraceLogger(object):
    current_logger = None

    def __init__(self, current_logger):
        self.current_logger = current_logger

    def __getattr__(self, item):
        if item == '__wrapped__':
            raise AttributeError
        return self.current_logger.exception


class SensitiveLogger(object):
    """
    Логгер, который проверяет, не пытаются ли залоггировать чувствительные данные
    !!!DEPRECATED!!!
    Использовать default_log и error_log
    """
    thread_local = threading.local()

    def __getattr__(self, item):
        if item == '__wrapped__':
            raise AttributeError
        return self.get_logger(item)

    def get_logger(self, log_name):
        def wrapper(*args, **kwargs):
            if 'extra' not in kwargs:
                kwargs['extra'] = {}

            kwargs['extra']['context'] = getattr(self.thread_local, 'context', {})

            if log_name in ['error', 'exception']:
                return getattr(error_log, 'exception')(*args, **kwargs)
            return getattr(default_log, log_name)(*args, **kwargs)

        return wrapper

    @contextmanager
    def fields(self, **kwargs):
        with log_context(**make_json_serializable(kwargs)):
            self.thread_local.context = kwargs
            yield self
            self.thread_local.context = None

    @contextmanager
    def name_and_fields(self, name, **kwargs):
        with self.fields(**kwargs):
            yield

    def trace(self):
        return self


class DirectoryLogger(logging.Logger):
    def makeRecord(self, name, level, fn, lno, msg, args, exc_info, func=None, extra=None, sinfo=None):
        from intranet.yandex_directory.src.yandex_directory.core.utils.ycrid_manager import ycrid_manager

        if extra is None:
            extra = {}

        extra['host_name'] = os.environ.get('DEPLOY_POD_PERSISTENT_FQDN', 'localhost')
        extra['appname'] = '%s-%s' % (
            os.environ.get('DEPLOY_STAGE_ID', ''),
            os.environ.get('DEPLOY_UNIT_ID', '').replace('_standalone', '')
        )
        ycrid, req_id = ycrid_manager.get()
        extra['ycrid'] = ycrid
        extra['request_id'] = req_id

        return super(DirectoryLogger, self).makeRecord(name, level, fn, lno, msg, args, exc_info, func, extra, sinfo)


class TSKVFormatter(logging.Formatter):
    _special_substitutions = [
        ('\\', '\\\\'),
        ('\t', r'\t'),
        ('\r', r'\r'),
        ('\n', r'\n'),
        ('\0', r'\0'),
    ]

    def format(self, record: logging.LogRecord) -> str:
        if self.usesTime():
            record.asctime = self.formatTime(record, self.datefmt)

        record.message = record.getMessage()
        if not hasattr(record, 'context'):
            record.context = {}

        if record.exc_info:
            # Кэшируем трейс, т.к. он сохраняется между логгерами.
            if not record.exc_text:
                record.exc_text = self.formatException(record.exc_info)
        if record.exc_text:
            if record.message[-1:] != r"\n":
                record.message += r"\n"
            record.message = record.message + record.exc_text

        return self._fmt % self.escape(record.__dict__)

    def escape(self, data):
        if isinstance(data, str):
            for search, replacement in self._special_substitutions:
                data = data.replace(search, replacement)
        elif isinstance(data, dict):
            new_data = {}
            for key, value in data.items():
                new_data[key] = self.escape(value)
            data = new_data
        elif isinstance(data, (list, tuple)):
            new_data = []
            for value in data:
                new_data.append(self.escape(value))
            if isinstance(data, tuple):
                data = tuple(new_data)
            else:
                data = new_data
        return data


log = SensitiveLogger()
logging.setLoggerClass(DirectoryLogger)
access_log = logging.getLogger('dir_access')
default_log = logging.getLogger('dir_default')
error_log = logging.getLogger('dir_error')
requests_log = logging.getLogger('dir_requests')
