# coding: utf-8
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


from intranet.yandex_directory.src.yandex_directory import app
from functools import (
    partial,
    wraps,
)
from retrying import retry

from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    url_join,
    log_service_response,
    get_user_data_from_blackbox_by_uid,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    OrganizationModel,
    organization_type,
)
from intranet.yandex_directory.src.yandex_directory.common import http_client


def skip_for_autotests(func):
    """ Ничего не делаем с диском в момент запуска автотестов и в стрессовом окружении"""
    @wraps(func)
    def wrapper(*args, **kwargs):
        if not app.config['DISK_BASE_URL'] and app.config['ENVIRONMENT'] == 'autotests':
            return True
        if app.config['ENVIRONMENT'] == 'stress':
            return True
        return func(*args, **kwargs)
    return wrapper


# делаем максимум 10 попыток
# с экспоненциальным увеличением интервала
# вплоть до 10 секунд:
# 1 2 4 8 10 10 10 10
#
# TODO: позднее, здесь надо будет сделать так, чтобы
# в случае если несколько ретраев не помогли, то задача
# перекладывалась в очередь
@skip_for_autotests
@retry(wait_exponential_multiplier=500,
       wait_exponential_max=10000,
       stop_max_attempt_number=10,
       retry_on_exception=retry_http_errors('disk'))
def _make_request(method, handle, **kwargs):
    base_url = app.config['DISK_BASE_URL']
    url = url_join(base_url, handle)
    headers = {'X-Ya-Service-Ticket': tvm.tickets['disk']}

    response = http_client.request(method, url, params=kwargs, headers=headers)

    log_service_response('disk', method, url, kwargs, response)
    response.raise_for_status()
    return response.json()


_get = partial(_make_request, 'get')
_post = partial(_make_request, 'post')


@skip_for_autotests
def is_user_active(uid):
    data = _get('user_check', uid=uid)
    return data.get('need_init') != '1'


@skip_for_autotests
def get_user_info(uid):
    return _get('user_info', uid=uid)


@skip_for_autotests
def is_paid(uid):
    if not is_user_active(uid):
        return False
    user_info = get_user_info(uid)
    return int(user_info.get('paid', 0)) != 0


@skip_for_autotests
def is_b2b_user(uid):
    if not is_user_active(uid):
        return False
    info = get_user_info(uid)
    if info.get('is_b2b'):
        return True
    return False


# Таск взамен старого activate_user
class ActivateUserDiskTask(Task):
    default_priority = 75

    @skip_for_autotests
    def do(self, org_id, uid):
        """
        Активация пользователя в диске.
        Если не активирован то активируем, если активирован то делаем b2b.
        :param org_id:
        :param uid:
        :return:
        """
        # для пользователей порталов диск не активируем
        if OrganizationModel(self.main_connection).get_organization_type(org_id) == organization_type.portal:
            return

        if not is_user_active(uid):
            log.info('Activating user in Yandex.Disk')
            params = dict(
                uid=uid,
                b2b_key=org_id,
            )

            # noemail = 1 - не отправляем новому пользователю сообщение про место на диске
            if app.config['DISK_NO_EMAIL']:
                params['noemail'] = 1

            _post(
                'user_init',
                **params
            )
        else:
            # // если пользователь уже есть в диске,
            # // но он не битубишный, сделаем его таким
            log.info('Making user b2b in Yandex.Disk')
            if not is_b2b_user(uid):
                _post(
                    'user_make_b2b',
                    uid=uid,
                    b2b_key=org_id,
                )


@skip_for_autotests
def deactivate_user(org_id, uid):
    with log.fields(org_id=org_id, uid=uid):
        if is_b2b_user(uid):
            params = {'uid': uid}

            # noemail = 1 - не отправляем новому пользователю сообщение про место на диске
            if app.config['DISK_NO_EMAIL']:
                params['noemail'] = 1

            log.info('Deactivating user')
            _post('user_reset_b2b', **params)


# Таск взамен старого async_user_remove
class RemoveUserFromDiskTask(Task):
    @skip_for_autotests
    def do(self, uid, org_id):
        log.info('Removing user from Yandex.Disk')
        # проверяем, что пользователь есть в паспорте и b2b
        if get_user_data_from_blackbox_by_uid(uid) and is_b2b_user(uid):
            return _get('async_user_remove', uid=uid)
