# -*- coding: utf-8 -*-
# https://wiki.yandex-team.ru/mail/pdd/fouras/

from functools import partial
from retrying import retry
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.common.utils import url_join
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError
from intranet.yandex_directory.src.yandex_directory.common import http_client


class FourasError(APIError):
    code = 'unknown_fouras_error'
    message = 'Error in Fouras API'
    description = 'Неизвестная ошибка сервиса управления DKIM.'


class FourasNotFoundError(FourasError):
    status_code = 404
    code = 'fouras_domain_not_found'
    message = 'Domain not found in Fouras'
    description = 'Домен не найден в сервисе управления DKIM'


def raise_on_errors(fouras_response):
    if fouras_response.status_code == 404:
        raise FourasNotFoundError()
    if fouras_response.status_code >= 400:
        with log.fields(error=fouras_response.text):
            log.error('Fouras returned error')
            raise FourasError()

    json_response = fouras_response.json()
    if json_response['status'] != 'ok':
        with log.fields(error=json_response['response']):
            log.error('Fouras returned error')
            raise FourasError()

    return json_response


@retry(stop_max_attempt_number=3, retry_on_exception=retry_http_errors('fouras'))
def _make_request(method, uri, get_params=None, post_params=None):
    ticket = tvm.tickets['fouras']
    headers = {
        'X-Ya-Service-Ticket': ticket,
    }
    fouras_url = app.config['FOURAS_HOST']
    if not fouras_url:
        raise RuntimeError('FOURAS base URL is unknown. Set FOURAS_HOST variable')

    url = url_join(fouras_url, uri, query_params=get_params)
    response = http_client.request(method, url, json=post_params, headers=headers)
    return raise_on_errors(response)


_get = partial(_make_request, 'get')
_post = partial(_make_request, 'post')


def domain_status(domain, selector=None):
    """
    {
       "response" : {
          "enabled" : false,
          "domain" : "music.yandex.ru",
          "changed" : true,
          "selector" : "mail",
          "public_key" : "v=DKIM1; k=rsa; t=s; p=..."
       "status" : "ok"
    }
    """
    params = {
        'domain': domain,
        'selector': selector or 'mail'
    }
    with log.fields(**params):
        response = _get('/domain/status', get_params=params)
        return response['response']


def domain_key_gen(domain, selector=None):
    """
    {
       "response" : {
          "public_key" : "v=DKIM1; k=rsa; t=s; p=..."
          "private_key": "....",
          "domain": "yandex.ru",
          "selector": "mail",
          "enabled": true
       },
       "status" : "ok"
    }
    """
    data = {
        'domain': domain,
        'selector': selector or 'mail'
    }
    with log.fields(**data):
        response = _post('/domain/key/gen', post_params=data)
        return response['response']


def get_or_gen_domain_key(domain_name):
    try:
        return domain_status(domain_name, 'mail')['public_key']
    except FourasNotFoundError:
        return domain_key_gen(domain_name, 'mail')['public_key']
