# coding: utf-8

import json


from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.common.utils import url_join, to_punycode
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common import http_client


class GendarmeError(APIError):
    code = 'gendarme_error'
    message = 'Error in Gendarme API'
    description = 'Ошибка сервиса, проверяющего MX записи.'


class GendarmeDomainNotFound(APIError):
    status_code = 404
    code = 'gendarme_domain_not_found'
    message = 'Domain not found in Gendarme'
    description = 'Домен не найден при проверке MX'


def raise_on_errors(response):
    """Проверяет есть ли ошибки, и если нет, то возвращает ответ сервера."""
    if response.status_code == 404:
        raise GendarmeDomainNotFound()

    if response.status_code >= 400:
        with log.fields(response=response.text):
            log.error('Gendarme error')
        raise GendarmeError()

    data = response.json()

    if data['status'] != 'ok':
        with log.fields(response=data):
            log.error('Gendarme error')
            raise GendarmeError()

    return data


def status(domain):
    """Возвращает информацию про домен из сервиса Жандарм."
       https://wiki.yandex-team.ru/mail/pdd/gendarme/
    """
    with log.fields(domain=domain):
        headers = {'X-Ya-Service-Ticket': tvm.tickets['gendarme']}
        url = url_join(
            app.config['GENDARME_URL'],
            '/domain/status',
            query_params=dict(name=to_punycode(domain)),
        )
        response = http_client.request('get', url, headers=headers)
        data = raise_on_errors(response)
        return data['response']


def recheck(domain, sync=False):
    with log.fields(domain=domain):
        headers = {
            'X-Ya-Service-Ticket': tvm.tickets['gendarme'],
            'Content-Type': 'application/json',
        }
        url = url_join(
            app.config['GENDARME_URL'],
            '/domain/recheck',
        )
        data = dict(name=to_punycode(domain), sync=sync)
        data = json.dumps(data)
        response = http_client.request('post', url, headers=headers, data=data)
        return raise_on_errors(response)['response']

