# encoding: UTF-8

from abc import ABCMeta
from abc import abstractmethod

from sqlalchemy.orm import Session
from typing import Any
from typing import Dict
from typing import Optional

from intranet.yandex_directory.src.yandex_directory.limits.models import OrganizationLimit
from intranet.yandex_directory.src.yandex_directory.meta.repositories import MetaOrganizationRepository


class OrganizationLimiter(object, metaclass=ABCMeta):
    """
    Интерфейс компонента обеспечивающего бизнес-логику лимитов организации.
    """

    @abstractmethod
    def get_all_limits(self, org_id):
        # type: (Any) -> Dict[OrganizationLimit, int]
        raise NotImplementedError

    @abstractmethod
    def get_limit(self, org_id, limit):
        # type: (int, OrganizationLimit) -> int
        raise NotImplementedError

    @abstractmethod
    def set_limit(self, org_id, limit, value):
        # type: (int, OrganizationLimit, int) -> None
        raise NotImplementedError

    @abstractmethod
    def reset_limit(self, org_id, limit):
        # type: (int, OrganizationLimit) -> None
        raise NotImplementedError


class DefaultOrganizationLimiter(OrganizationLimiter):
    """
    Реализация по умолчанию компонента обеспечивающего бизнес-логику лимитов
    организации.

    Лимиты хранятся в поле ``limits`` модели ``MetaOrganization``.
    """
    BLANK_ON_DEFAULT = {
        'users_limit',
    }

    def __init__(
            self,
            repository,  # type: MetaOrganizationRepository
            defaults,  # type: Dict[OrganizationLimit, int]
    ):
        self._repository = repository

        for limit in list(OrganizationLimit.__members__.values()):
            if limit not in defaults:
                raise ValueError('Missing default value for %s' % limit)

        self._defaults = defaults

    def get_all_limits(self, org_id, session=None):
        # type: (Any, Optional[Session]) -> Dict[OrganizationLimit, int]
        meta_org = self._repository.get(org_id, session=session)
        return {
            limit: meta_org.limits.get(
                limit,
                self._defaults[limit] if limit not in self.BLANK_ON_DEFAULT
                else None
            )
            for limit in list(OrganizationLimit.__members__.values())
        }

    def get_limit(self, org_id, limit, session=None):
        # type: (Any, OrganizationLimit, Optional[Session]) -> int
        meta_org = self._repository.get(org_id, session=session)
        return meta_org.limits.get(limit, self._defaults[limit])

    def set_limit(self, org_id, limit, value, session=None):
        # type: (Any, OrganizationLimit, int, Optional[Session]) -> None
        meta_org = self._repository.get(org_id, session=session)
        meta_org.limits[limit] = value
        self._repository.save(meta_org, session=session)

    def reset_limit(self, org_id, limit, session=None):
        # type: (Any, OrganizationLimit, Optional[Session]) -> None
        meta_org = self._repository.get(org_id, session=session)
        meta_org.limits.pop(limit, None)
        self._repository.save(meta_org, session=session)
