# encoding: UTF-8

from flask import Blueprint
from flask import g
from flask import jsonify
from flask import request
from flask.views import MethodView
from marshmallow import ValidationError
from sqlalchemy.orm import Session

from intranet.yandex_directory.src.yandex_directory.auth.decorators import internal
from intranet.yandex_directory.src.yandex_directory.common.compat import (
    permission_required_compat,
    scopes_required_compat,
)
from intranet.yandex_directory.src.yandex_directory.common.components import component_registry
from intranet.yandex_directory.src.yandex_directory.common.marshmallow import PublicValidationError
from intranet.yandex_directory.src.yandex_directory.core.models import SupportActionMetaModel
from intranet.yandex_directory.src.yandex_directory.core.models.action import SupportActions
from intranet.yandex_directory.src.yandex_directory.core.permission import common_internal_permission
from intranet.yandex_directory.src.yandex_directory.core.permission import support_internal_permission
from intranet.yandex_directory.src.yandex_directory.limits.components import OrganizationLimiter
from intranet.yandex_directory.src.yandex_directory.limits.dto import LimitsSchema
from intranet.yandex_directory.src.yandex_directory.limits.dto import (
    UpdateLimitsAdminSchema,
    UpdateLimitsSchema
)
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    requires,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import (
    scope,
)

org_limits_bp = Blueprint('limits', __name__)


class BaseLimitsView(MethodView):
    @property
    def organization_limiter(self):
        # type: () -> OrganizationLimiter
        """
        Shortcut для доступа к лимитам организаций
        """
        return component_registry().organization_limiter

    @property
    def meta_session(self):
        # type: () -> Session
        """
        Shortcut для доступа к мета-сесссии
        """
        return component_registry().meta_session

    @internal
    @permission_required_compat([
        common_internal_permission.organizations_limits_view,
    ])
    @requires(org_id=False, user=False)
    def get(self, org_id):
        """
        Возвращает список всех лимитов организации.
        """
        limits = self.organization_limiter.get_all_limits(org_id)
        data = LimitsSchema().dump(limits)
        return jsonify(data)

    @internal
    @scopes_required_compat([scope.manage_limits])
    @requires(org_id=False, user=False)
    def patch(self, org_id):
        """
        Обновляет заданные лимиты организации. Если в качестве нового значения
        передан ``null`` то лимит сбрасывается на значение по умолчанию.
        """
        content = request.get_json(force=True)
        try:
            data = UpdateLimitsSchema().load(content)
        except ValidationError as exc:
            errors = exc.messages
            raise PublicValidationError(errors)
        else:
            for limit, value in data['limits'].items():
                if value is None:
                    self.organization_limiter.reset_limit(org_id, limit)
                else:
                    self.organization_limiter.set_limit(org_id, limit, value)
            return self.get(org_id)


class AdminOrganizationLimitsView(BaseLimitsView):

    @internal
    @permission_required_compat([
        support_internal_permission.change_organization_limits,
    ])
    @requires(org_id=False, user=False)
    def patch(self, org_id):
        """
        Обновляет заданные лимиты организации. Если в качестве нового значения
        передан ``null`` то лимит сбрасывается на значение по умолчанию.
        """
        content = request.get_json(force=True)
        try:
            data = UpdateLimitsAdminSchema().load(content)
        except ValidationError as exc:
            errors = exc.messages
            raise PublicValidationError(errors)
        else:
            for limit, value in data['limits'].items():
                if value is None:
                    self.organization_limiter.reset_limit(org_id, limit)
                else:
                    self.organization_limiter.set_limit(org_id, limit, value)

            model = SupportActionMetaModel(self.meta_session.connection())
            model.create(
                org_id=org_id,
                name=SupportActions.change_organization_limits,
                author_id=g.user.passport_uid,
                object_value=self.organization_limiter.get_all_limits(org_id),
                object_type='organization_limits',
                comment=data['comment'],
            )

            return self.get(org_id)


org_limits_bp.add_url_rule(
    '/admin/organizations/<int:org_id>/limits/',
    view_func=AdminOrganizationLimitsView.as_view('admin_organization_limits'),
)
org_limits_bp.add_url_rule(
    '/organizations/<int:org_id>/limits/',
    view_func=BaseLimitsView.as_view('organization_limits'),
)
