import os
import typing
import json
import tvmauth
import logging
from concurrent.futures import TimeoutError

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import error_log, default_log
import kikimr.public.sdk.python.persqueue.auth as auth
import kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api as pqlib
from kikimr.public.sdk.python.persqueue.errors import SessionFailureResult

from intranet.yandex_directory.src.yandex_directory import app


class LogbrokerException(Exception):
    """
    Exception handler for Logbroker errors
    """

    def __init__(self, message: str, status: str = "") -> None:
        self.message = message
        self.status = status


class LogbrokerClient(object):
    """
    Client for Logbroker based on pqlib SDK
    https://logbroker.yandex-team.ru/docs/how_to/develop_with_python
    """

    def __init__(self,
                 logbroker_endpoint: str = settings.LOGBROKER_ENDPOINT,
                 logbroker_port: int = settings.LOGBROKER_PORT,
                 timeout: int = 15, tvm_secret: str = settings.TVM_SECRET,
                 debug: bool = False):
        self.endpoint = logbroker_endpoint
        self.port = logbroker_port
        self.timeout = timeout

        tvm_secret = tvm_secret or os.environ.get("TVM_SECRET", "")
        if tvm_secret:
            self.tvm_client = tvmauth.TvmClient(tvmauth.TvmApiClientSettings(
                self_tvm_id=settings.DIRECTORY_TVM_CLIENT_ID,
                self_secret=tvm_secret,
                dsts={'logbroker': settings.LOGBROKER_TVM_ID},
            ))
            self.credentials = auth.TVMCredentialsProvider(self.tvm_client,
                                                           destination_alias="logbroker",)
        else:
            error_log.warning("TVM_SECRET wasn't set and environ variable TVM_SECRET wasn't found")
            self.credentials = None
        self.api = self.prepare_api()
        self.prepare_logging(debug)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()

    @staticmethod
    def prepare_logging(debug):
        """
        Prepare logging
        """
        if debug:
            loglevel = logging.DEBUG
        else:
            loglevel = logging.INFO
        logging.basicConfig(level=loglevel)
        logging.getLogger("kikimr.public.sdk.python.persqueue").setLevel(logging.WARNING)

    def prepare_api(self) -> pqlib.PQStreamingAPI:
        """
        Create PQStreamingAPI instance
        """
        api = pqlib.PQStreamingAPI(self.endpoint, self.port)
        api_start_future = api.start()
        api_start_future.result(timeout=self.timeout)
        return api

    def stop(self) -> None:
        """
        Call stop method for PQStreamingAPI
        """
        default_log.info("Stopping LogbrokerClient")
        self.api.stop()

    def get_consumer(self, consumer_name: str, topic: str,
                     json_encoder: typing.Optional[typing.Type[json.JSONEncoder]] = None):
        """
        Return LogbrokerConsumer instance for given source and topic

        :param consumer_name: Logbroker consumer name
        :param topic: Logbroker topic to write in
        :param json_encoder: JSON encoder for message decoding
        :return: LogbrokerConsumer
        """
        return LogbrokerConsumer(client=self, consumer_name=consumer_name, topic=topic, json_encoder=json_encoder)


class LogbrokerConsumer(object):
    """
    Logbroker Consumer
    """
    def __init__(self, client: LogbrokerClient, consumer_name: str, topic: str,
                 timeout: int = app.config['LOGBROKER_READ_TIMEOUT'],
                 json_encoder: typing.Optional[typing.Type[json.JSONEncoder]] = None) -> None:
        self.client = client
        self.topic = topic
        self.consumer_name = consumer_name
        self.timeout = timeout
        self.json_encoder = json_encoder
        self.consumer = self.prepare_consumer()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()

    def stop(self) -> None:
        default_log.info(f"Stop the consumer for topic {self.topic} and consumer_name {self.consumer_name}")
        self.consumer.stop()

    def commit(self, cookies) -> None:
        default_log.info("Committing cookies")
        for cookie in cookies:
            self.consumer.commit(cookie)

    def prepare_consumer(self) -> pqlib.PQStreamingConsumer:
        """
        Prepare consumer
        """
        configurator = pqlib.ConsumerConfigurator(
            self.topic,
            self.consumer_name,
            max_time_lag_ms=app.config['LOGBROKER_READ_MAX_LAG'] * 100 if app.config['LOGBROKER_READ_MAX_LAG'] else None,
            max_count=app.config['LOGBROKER_READ_TOTAL_MESSAGES_IN_REQUEST'])
        consumer = self.client.api.create_consumer(configurator, credentials_provider=self.client.credentials)
        default_log.info("Starting Consumer")
        start_future = consumer.start()
        result = start_future.result(timeout=self.timeout)

        if not isinstance(result, SessionFailureResult):
            if result.HasField("init"):
                default_log.info(f"Consumer start result was: {result}")
            else:
                raise LogbrokerException(f"Consumer failed to start with error {result}")
        else:
            raise LogbrokerException(f"Error occurred on start of consumer: {result}")
        default_log.info("Consumer started")
        return consumer

    def read(self, total_messages_expected: int = 1) -> typing.List[pqlib.ConsumerMessage]:
        """
        Read messages from topic
        :param total_messages_expected: Messages number to read from LB topic
        :return: list with ConsumerMessage instances
        """
        all_data = []
        result_cookies = []
        default_log.info("Read data")
        while total_messages_expected > 0:
            try:
                result = self.consumer.next_event().result(timeout=self.timeout)
            except TimeoutError:
                raise LogbrokerException(f"Failed to get any messages from topic {self.topic}")
            if result.type == pqlib.ConsumerMessageType.MSG_DATA:
                for batch in result.message.data.message_batch:
                    for message in batch.message:
                        all_data.append(message)
                        total_messages_expected -= 1
                        if total_messages_expected == 0:
                            self.consumer.reads_done()
                result_cookies.append(result.message.data.cookie)
        default_log.info(f"Reads done. Totally read {len(all_data)} messages")
        return all_data, result_cookies
