import time
import zlib

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.logbroker.client import LogbrokerClient, LogbrokerException
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import error_log, default_log
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.utils.ycrid_manager import ycrid_manager


class AbstractConsumer(object):
    """
    Abstract class to work with Logbroker consumer.
    """
    def __init__(self, lb_topic: str, lb_consumer: str,
                 codec: str = 'gzip',
                 lb_endpoint: str = settings.LOGBROKER_ENDPOINT,
                 lb_port: int = settings.LOGBROKER_PORT):
        self.lb_endpoint = lb_endpoint
        self.lb_port = lb_port
        self.lb_topic = lb_topic
        self.lb_consumer = lb_consumer
        self.codec = codec

    @staticmethod
    def _tskv_to_dict(tskv_row: bytes):
        """
        Convert TSKV row to dict
        """
        return {k: v for k, v in filter(lambda x: len(x) == 2, map(lambda x: x.split(b'=', 1), tskv_row.split(b'\t')))}

    def _handle_rows(self, message):
        """
        Main method to handle Logbroker messages
        """
        pass

    def run(self):
        """
        Run Logbroker consumer for given topic.
        for every topic because we use partitions for parallel running tasks.
        """
        with LogbrokerClient(logbroker_endpoint=self.lb_endpoint,
                             logbroker_port=self.lb_port) as lb_client:
            time_interval = app.config['LOGBROKER_READ_INTERVAL_MIN']
            while True:
                with lb_client.get_consumer(consumer_name=self.lb_consumer,
                                            topic=self.lb_topic) as consumer:
                    try:
                        ycrid_manager.generate("consumer")
                        messages, result_cookies = consumer.read(total_messages_expected=app.config['LOGBROKER_READ_TOTAL_MESSAGES'])
                        default_log.info(f"Got messages from topic {self.lb_topic}")
                        for message in messages:
                            # If message have codec GZIP, need to decompress it
                            if self.codec == 'gzip':
                                message = zlib.decompress(message.data, zlib.MAX_WBITS | 16)
                            self._handle_rows(message)
                        consumer.commit(result_cookies)

                        if not messages:
                            default_log.info(f"Sleep {time_interval}")
                            time.sleep(time_interval)
                            time_interval = min(app.config['LOGBROKER_READ_INTERVAL_MAX'], time_interval * app.config['LOGBROKER_READ_INTERVAL_MULTIPLIER'])
                        else:
                            time_interval = app.config['LOGBROKER_READ_INTERVAL_MIN']
                    except LogbrokerException as exc:
                        error_log.warning(
                            f"Can't read messages from topic {self.lb_topic}, probably there are no new messages: {exc}"
                        )
                    except Exception as exc:
                        error_log.exception(f"Got unhandled exception {exc}", exc_info=True)
