# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
import requests
import boto3

from botocore import UNSIGNED
from botocore.client import Config


from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    url_join,
)
from intranet.yandex_directory.src.yandex_directory.mds.exceptions import (
    raise_mds_error,
    MDSUnavailableError,
)
from intranet.yandex_directory.src.yandex_directory.common import http_client


class MdsApiClient(object):
    def __init__(self):
        self.mds_url = app.config['MDS_WRITE_API']
        self.mds_namespace = app.config['MDS_NAMESPACE']
        self.default_params = {}
        self.method_templates = {
            # Загрузка изображения на mds
            # https://wiki.yandex-team.ru/mds/avatars/#put
            'put': '/put-' + self.mds_namespace + '/{image_name}/',
        }
        self.headers = {}

    def _write_log(self, url, data, response):
        try:
            resp = response.json()
        except ValueError:
            resp = response.text
        fields = {
            'url': url,
            'data': data,
            'response': resp,
            'response_code': response.status_code,
        }
        with log.name_and_fields('mds-requests', **fields):
            log_func = log.error if response.status_code >= 400 else log.info
            log_func('MDS-request')

    def _make_post_request(self, url, files, headers=None, log_data=True, **kwargs):
        if headers is None:
            headers = self.headers

        response = http_client.request('post', url, files=files, headers=headers, **kwargs)
        if not log_data:
            files = '[REMOVED]'
        self._write_log(url, files, response)

        response_data = response.json()

        if response.status_code != 200:
            raise_mds_error(response_data)

        return response_data


    def _make_get_request(self, url, data, headers=None, log_data=True, **kwargs):
        if headers is None:
            headers = self.headers

        response = http_client.request('get', url, params=data, headers=headers, **kwargs)
        if not log_data:
            data = '[REMOVED]'
        self._write_log(url, data, response)


        response_data = response.json()

        if response.status_code != 200:
            raise_mds_error(response_data)

        return response_data

    def _construct_url(self, part_url, query_params=None):
        params = self.default_params.copy()
        if query_params:
            params.update(query_params)
        return url_join(self.mds_url, part_url, query_params=params)

    def _upload_image_by_file(self, image_id, file_img):
        """Загружаем файл изображения.
        """
        url = self._construct_url(
            self.method_templates['put'].format(
                image_name=image_id,
            )
        )

        files = {'file': file_img}
        return self._make_post_request(url, files=files)

    def _upload_image_by_url(self, image_id, url_img):
        """Загружаем изображение по url.
        """
        url = self._construct_url(
            self.method_templates['put'].format(
                image_name=image_id,
            )
        )

        data = {'url': url_img}

        return self._make_get_request(url, data=data)

    def upload_image(self, image_id, file_img, url_img):
        """Загружаем изображение в mds.
        """

        try:
            if file_img:
                result = self._upload_image_by_file(image_id, file_img)
            elif url_img:
                result = self._upload_image_by_url(image_id, url_img)
            else:
                result = None
            # Если MDS выбросит какие-то другие исключения, связанные
            # с форматом запроса и тому подобным, то это будет MDSError,
            # который наследуется от APIError и будет корректно обработан
            # диспатчером.
        except (requests.exceptions.ConnectionError, requests.exceptions.Timeout):
            log.trace().error('Unable to connect to MDS.')
            raise MDSUnavailableError()


        return result


class MdsS3ApiClient:
    def __init__(self):
        self.client = boto3.client(
            service_name='s3',
            endpoint_url=app.config['MDS_S3_API_URL'],
            verify=False,
            config=Config(signature_version=UNSIGNED)
        )

    def __getattr__(self, item):
        if item == '__wrapped__':
            raise AttributeError
        return getattr(self.client, item)

    def get_object_content(self, bucket, object_id):
        response = self.client.get_object(
            Bucket=bucket,
            Key=object_id,
        )
        return response['Body'].read()
