# encoding: UTF-8

import sys
from json import JSONDecoder

from flask_script import Option
from marshmallow import Schema
from marshmallow.fields import Boolean
from marshmallow.fields import Integer
from marshmallow.fields import String
from sqlalchemy.orm import Session
from sqlalchemy.orm.exc import NoResultFound
from marshmallow.exceptions import ValidationError

from intranet.yandex_directory.src.yandex_directory.common.commands.base import BaseCommand
from intranet.yandex_directory.src.yandex_directory.common.components import component_registry
from intranet.yandex_directory.src.yandex_directory.common.datatools import flatten
from intranet.yandex_directory.src.yandex_directory.meta.components import get_meta_session
from intranet.yandex_directory.src.yandex_directory.meta.repositories import (
    RegistrarRepository,
    DomainTokenRepository,
)
from intranet.yandex_directory.src.yandex_directory.meta.models import (
    Registrar,
    DomainToken,
)


class RegistrarSchema(Schema):
    """
    Схема для парсинга регистраторов в формате JSON.
    """

    id = Integer(required=True)
    new = Boolean(attribute='is_new', required=True)
    admin_id = Integer(required=True)
    password = String(required=True)
    name = String(required=True)
    iv = String(required=True)
    plength = Integer(required=True)
    validate_domain_callback = String(required=True)
    delete_domain_callback = String(required=True)
    added_domain_callback = String(required=True)
    verified_domain_callback = String(required=True)


class DomainTokenSchema(Schema):
    """
    Схема для парсинга регистраторов в формате JSON.
    """
    pdd_version = String(required=True)
    admin_id = Integer(required=True)
    domain = String(required=True)
    token = String(required=True)


class Command(BaseCommand):
    """
    Импортирует регистраторов экспортированных из YT в JSON-формате.
    Поддерживает dry-run режим.
    """

    name = 'import-registrars'

    option_list = [
        Option('--dry-run', '-d', dest='dry_run', action='store_true'),
        Option('--domain-tokens', '-t', dest='domain_tokens', action='store_true'),
    ]

    @property
    def meta_session(self):
        # type: (...) -> Session
        """
        Shortcut для доступа к мета-сесссии
        """
        return component_registry().meta_session

    @property
    def domain_token_repository(self):
        # type: (...) -> DomainTokenRepository
        """
        Shortcut для доступа к репозиторию токенов доменов
        """
        return component_registry().domain_token_repository

    @property
    def registrar_repository(self):
        # type: (...) -> RegistrarRepository
        """
        Shortcut для доступа к репозиторию регистраторов
        """
        return component_registry().registrar_repository

    def _read(self, f, schema):
        """
        Парсит JSON-строки из стаднтартного потока ввода.
        """
        decoder = JSONDecoder()
        for line_no, line in enumerate(f, 1):
            line = line.strip()
            if not line:
                continue
            try:
                line_data = decoder.decode(line)
            except ValueError:
                print('Failed to parse line #%d.' % line_no, file=sys.stderr)
            else:
                try:
                    data = schema.load(line_data)
                except ValidationError as exc:
                    errors = exc.messages
                    for field, messages in flatten(errors):
                        for message in messages:
                            msg = 'Invalid data in line #%d: %s: %s' % (
                                line_no, field, message
                            )
                            print(msg, file=sys.stderr)
                else:
                    yield data

    def import_registrars(self):

        for raw_registrar in self._read(sys.stdin, RegistrarSchema()):
            pdd_id = raw_registrar['id']
            pdd_version = 'new' if raw_registrar['is_new'] else 'old',
            try:
                # Ищем регистратора по идентификатору ПДД
                registrar = self.registrar_repository.find_one_by_pdd_id(
                    pdd_id,
                    pdd_version,
                )
            except NoResultFound:
                # Если не нашли создаем нового
                registrar = Registrar()
                self.meta_session.add(registrar)

            registrar.pdd_id = pdd_id
            registrar.pdd_version = pdd_version
            registrar.name = raw_registrar['name']
            registrar.admin_id = raw_registrar['admin_id']
            registrar.password = raw_registrar['password']
            registrar.iv = raw_registrar['iv']
            registrar.plength = raw_registrar['plength']

            registrar.validate_domain_url = \
                raw_registrar['validate_domain_callback']

            registrar.domain_added_callback_url = \
                raw_registrar['added_domain_callback']

            registrar.domain_verified_callback_url = \
                raw_registrar['verified_domain_callback']

            registrar.domain_deleted_callback_url = \
                raw_registrar['delete_domain_callback']

    def import_domain_tokens(self):

        for raw_domain_token in self._read(sys.stdin, DomainTokenSchema()):
            pdd_version = raw_domain_token['pdd_version']
            domain = raw_domain_token['domain']
            admin_id = raw_domain_token['admin_id']
            domain_token = self.domain_token_repository.find_one_by_pdd_info(
                admin_id,
                domain,
                pdd_version,
            )

            if not domain_token:
                # Если не нашли создаем нового
                domain_token = DomainToken()
                self.meta_session.add(domain_token)

            domain_token.pdd_version = pdd_version
            domain_token.domain = domain
            domain_token.admin_id = admin_id
            domain_token.token = raw_domain_token['token']

    def run(self, dry_run, domain_tokens):
        # Форсим создание мета-сессии в режиме для записи
        get_meta_session(is_modifying=True)

        if domain_tokens:
            self.import_domain_tokens()
        else:
            self.import_registrars()

        if dry_run:
            # В режиме dry run откатываем все внесенные изменения
            self.meta_session.rollback()
