# encoding: UTF-8

from sqlalchemy.orm import Session
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.orm import joinedload

from intranet.yandex_directory.src.yandex_directory.common.sqlalchemy import (
    CRUDRepository,
)
from intranet.yandex_directory.src.yandex_directory.meta.models import (
    MetaOrganization,
    MetaUser,
    Registrar,
    DomainToken
)


class MetaOrganizationRepository(CRUDRepository):
    """
    Репозиторий мета-организаций.
    """

    def __init__(self, session_provider):
        super(MetaOrganizationRepository, self).__init__(
            MetaOrganization,
            session_provider,
        )


class MetaUserRepository(CRUDRepository):
    """
    Репозиторий мета-организаций.
    """

    def __init__(self, session_provider):
        super(MetaUserRepository, self).__init__(
            MetaUser,
            session_provider,
        )

    def get_org_ids_with_shard_numbers(self, user_id, limit=None, offset=None, session=None):
        """Возвращает пользователя с подтянуттой информацией про шарды.

        Возвращаются только организации в которых человек не уволен.
        """
        query = self._query(session) \
                    .filter(MetaUser.id == user_id) \
                    .order_by(MetaUser.org_id) \
                    .options(joinedload('organization'))

        if limit is not None:
            query = query.limit(limit)

        if offset is not None:
            query = query.offset(offset)

        return query.all()


class RegistrarRepository(CRUDRepository):
    """
    Репозиторий регистраторов.
    """

    def __init__(self, session_provider):
        super(RegistrarRepository, self).__init__(
            Registrar,
            session_provider,
        )

    def find_one_by_pdd_id(self, pdd_id, pdd_version, session=None):
        # type: (str, Session) -> Registrar
        """
        Ищет регистратора по его идентификатору ПДД. В случае отсутсвия или
        наличия более одного регистратора вызывает соответсвующие исключения.
        """
        return self._query(session).filter(
            Registrar.pdd_id == pdd_id,
            Registrar.pdd_version == pdd_version,
        ).one()

    def find_one_by_id(self, id, session=None):
        # type: (str, Session) -> Registrar
        """
        Ищет регистратора по его идентификатору. В случае отсутсвия или
        наличия более одного регистратора вызывает соответсвующие исключения.
        """
        return self._query(session).filter(Registrar.id == id).one()

    def find_one_by_token(self, token, session=None):
        # type: (token, Session) -> Registrar
        """
        Ищет регистратора по его токену. В случае отсутсвия или
        наличия более одного регистратора вызывает соответсвующие исключения.
        """
        try:
            return self._query(session).filter(Registrar.token == token).one()
        except NoResultFound:
            pass

    def find_one_by_admin_id(self, admin_id, pdd_version, session=None):
        # type: (admin_id, pdd_versio, Session) -> Registrar
        """
        Ищет регистратора по uid администратора. В случае отсутсвия или
        наличия более одного регистратора вызывает соответсвующие исключения.
        """
        try:
            return self._query(session).filter(
                Registrar.admin_id == admin_id,
                Registrar.pdd_version == pdd_version
            ).one()
        except NoResultFound:
            pass


class DomainTokenRepository(CRUDRepository):
    """
    Репозиторий токенов доменов.
    """

    def __init__(self, session_provider):
        super(DomainTokenRepository, self).__init__(
            DomainToken,
            session_provider,
        )

    def find_one_by_pdd_info(self, admin_id, domain, pdd_version, session=None):
        # type: (int, str, str, Session) -> DomainToken
        try:
            return self._query(session).filter(
                DomainToken.admin_id == admin_id,
                DomainToken.pdd_version == pdd_version,
                DomainToken.domain == domain,
            ).one()
        except NoResultFound:
            pass

    def find_one_by_id(self, id, session=None):
        # type: (str, Session) -> DomainToken
        """
        Ищет домен по его идентификатору. В случае отсутсвия или
        наличия более одного домена вызывает соответсвующие исключения.
        """
        return self._query(session).filter(DomainToken.id == id).one()

    def find_one_by_token(self, token, session=None):
        # type: (token, Session) -> DomainToken
        """
        Ищет домен по его токену. В случае отсутсвия или
        наличия более одного домена вызывает соответсвующие исключения.
        """
        try:
            return self._query(session).filter(DomainToken.token == token).one()
        except NoResultFound:
            pass

    def find_one_by_token_and_pdd_version(self, token, pdd_version, session=None):
        # type: (str, str, Session) -> DomainToken
        try:
            return self._query(session).filter(
                DomainToken.token == token,
                DomainToken.pdd_version == pdd_version,
            ).one()
        except NoResultFound:
            pass
