# -*- coding: utf-8 -*-

import re

import requests

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    url_join,
    get_localhost_ip_address,
    force_text,
    json_error,
    remove_sensitive_data,
    get_domain_info_from_blackbox,
    login_to_punycode,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    is_outer_uid,
    is_cloud_uid,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import is_yandex_team_uid
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.passport import exceptions
from intranet.yandex_directory.src.yandex_directory.common import http_client

LOGIN_VALIDATE_RE = re.compile(r'[a-zA-Z0-9._-]{1,40}')


def get_exception_by_name(name):
    """Возвращает класс exception по имени.
       Например 'domain.not_found' => 'DomainNotFound'
    """
    error_name = name.title().replace('.', '').replace('_', '')
    return getattr(exceptions, error_name, exceptions.PassportException)


def raise_exception_if_errors(json_response):
    """Вспомогательная функция, которая проверяет ответ от Паспорта и кидает исключение
       если в ответе есть ошибка.
    """

    if json_response.get('status') == 'error' and json_response.get('errors'):
        error = json_response['errors'][0]
        exception_cls = get_exception_by_name(error)
        raise exception_cls(error=error)


class PassportApiClient(object):
    def __init__(self):
        self.passport_url = app.config['PASSPORT_API']
        self.default_params = {
            'consumer': 'directory',
        }
        self.method_templates = {
            # Create Passport-account:
            # https://wiki.yandex-team.ru/passport/api/bundle/registration/pdd/
            'account_add': '/1/bundle/account/register/pdd/',

            # Edit Passport-account:
            # https://wiki.yandex-team.ru/passport/api/bundle/changeaccount/#izmenitpersonalnyedannyebandlovajaversija
            'account_edit': '/1/bundle/account/person/',

            # Delete Passport-account:
            # https://wiki.yandex-team.ru/passport/python/api/bundle/deleteaccount/
            'account_delete': '/1/bundle/account/{uid}/',

            # Добавляем домен в паспорте
            # https://wiki.yandex-team.ru/passport/api/bundle/mdapi/#sozdaniedomena
            'domain_add': '/1/bundle/pdd/domain/',

            # Изменяем свойства домена (пока изменяем только organization_name)
            # https://wiki.yandex-team.ru/passport/python/api/bundle/account/#izmeneniesvojjstvakkaunta
            'domain_edit': '/1/bundle/pdd/domain/%s/',

            # меняем мастер домен
            # https://wiki.yandex-team.ru/passport/api/bundle/mdapi/#obmendomenasodnimizegoaliasov
            'change_master_domain': '/1/bundle/pdd/domain/%s/alias/%s/make_master/',

            # https://wiki.yandex-team.ru/passport/api/bundle/account/flushpdd/
            'change_password_submit': '/1/account/%s/flush_pdd/submit/',

            # https://wiki.yandex-team.ru/passport/api/bundle/account/flushpdd/
            'change_password_commit': '/1/account/%s/flush_pdd/commit/',

            # https://wiki.yandex-team.ru/passport/api/bundle/mdapi/#sozdaniealiasadljapolzovatelja
            'add_alias': '/1/account/{uid}/alias/pddalias/{alias}/',

            # https://wiki.yandex-team.ru/passport/api/bundle/mdapi/#udaleniealiasapolzovatelja
            'delete_alias': '/1/account/{uid}/alias/pddalias/{alias}/',

            # блокировка/разблокировка пользователя (меняем ему is_enabled)
            # https://wiki.yandex-team.ru/passport/api/bundle/changeaccount/#izmenitsvojjstvaakkaunta)
            'block_user': '/1/account/%s/',

            # Загрузка аватарки по UID
            # https://wiki.yandex-team.ru/passport/api/bundle/changeavatar/#zagruzkaavatarkipouidsessionnojjkukeilitokenubeztreka
            'change_avatar': '/2/change_avatar/',

            # Изменяем настройки аккаунта
            # https://wiki.yandex-team.ru/passport/api/bundle/changeaccount/#izmenitnastrojjkiakkauntazablokirovannostfaktsovmestnogoispolzovanijavkljuchjonnostparolejjprilozhenijjit.p
            'change_options': '/2/account/%s/options/',

            # Подписка аккаунтов на сиды.
            # Пока используется для того, чтобы "принимать" пользовательское
            # соглашение при создании роботных аккаунтов.
            #
            # Документация:
            # https://wiki.yandex-team.ru/passport/api/bundle/managesids/#sozdanieiliizmeneniepodpiski
            'subscription': '/1/account/{uid}/subscription/{service_slug}/',

            # Добавить домен как алиас
            # https://wiki.yandex-team.ru/passport/api/bundle/mdapi/#sozdaniealiasadljadomena
            'domain_alias_add': '/1/bundle/pdd/domain/{domain_id}/alias/',

            # Удалить алиас домена
            # https://wiki.yandex-team.ru/passport/api/bundle/mdapi/#udaleniealiasadomena
            'domain_alias_delete': '/1/bundle/pdd/domain/{domain_id}/alias/{alias_id}/',

            # Валидируем логины (label групп и отделов)
            # https://wiki.yandex-team.ru/passport/api/bundle/validate/#proveritloginbundleversija
            'validate_login': '/1/bundle/validate/login/',

            # Валидируем display name
            # https://wiki.yandex-team.ru/passport/api/bundle/validate/#validacijaimenifamiliiidisplayname
            'validate_display_name': '/1/bundle/validate/display_name/',

            # Валидируем firstname
            # https://wiki.yandex-team.ru/passport/api/bundle/validate/#validacijaimenifamiliiidisplayname
            'validate_firstname': '/1/bundle/validate/firstname/',

            # Валидируем lastname
            # https://wiki.yandex-team.ru/passport/api/bundle/validate/#validacijaimenifamiliiidisplayname
            'validate_lastname': '/1/bundle/validate/lastname/',

            # Валидируем пароль
            # https://wiki.yandex-team.ru/passport/api/bundle/validate/#proveritparolbandlovajaversija
            'validate_password': '/1/bundle/validate/password/',

            # Создаем трек
            # https://wiki.yandex-team.ru/passport/api/bundle/track/#sozdattrek
            'create_track': '/1/track/',

            # Валидируем натуральный домен в Паспорте
            # https://wiki.yandex-team.ru/passport/api/bundle/validate/#provalidirovatpdd-domen
            'validate_natural_domain': '/1/bundle/validate/domain/',

            # Валидируем коннектный домен в Паспорте
            # https://wiki.yandex-team.ru/passport/api/bundle/validate/#provalidirovatdomendirektorii
            'validate_connect_domain': '/1/bundle/validate/directory_domain/',

            # Удалить домен из паспортной базы
            # https://wiki.yandex-team.ru/passport/api/bundle/mdapi/#udaleniedomena
            'domain_delete': '/1/bundle/pdd/domain/{domain_id}/'
        }
        self.headers = {
            'Ya-Client-Host': 'yandex.ru',
            'Ya-Consumer-Client-Ip': get_localhost_ip_address(),
            'Ya-Consumer-Client-Scheme': 'https',
        }

    def _write_log(self, url, data, status_code, content, trace=False):
        if isinstance(data, dict):
            data = remove_sensitive_data(data, secret_params=['password', 'password_hash'])

        fields = {
            'url': url,
            'data': data,
            'response': content,
            'response_code': status_code,
        }
        with log.name_and_fields('passport-requests', **fields):
            log_func = log

            if trace:
                log_func = log_func.trace()

            log_func = log_func.error if status_code >= 400 else log_func.info

            log_func('Response from Passport')

    def _construct_url(self, part_url, query_params=None):
        params = self.default_params.copy()
        if query_params:
            params.update(query_params)
        return url_join(self.passport_url, part_url, query_params=params)

    def account_add(self, domain, user_data):
        data = {
            'login': login_to_punycode(user_data['login']),
            'domain': to_punycode(domain),
        }

        optional_params = ('firstname', 'lastname', 'gender', 'language', 'birthday')
        for param in optional_params:
            value = user_data.get(param)
            if value:
                data[param] = value
        if not data.get('gender'):
            data['gender'] = 'male'

        password = user_data.get('password')
        password_hash = user_data.get('password_hash')

        if password is not None:
            data['password'] = password
        elif password_hash is not None:
            data['password_hash'] = password_hash
        else:
            data['no_password'] = 1
            data['is_maillist'] = 1

        url = self._construct_url(self.method_templates['account_add'])
        json_response = self._make_post_request(url, data)
        return json_response.get('uid')

    def account_edit(self, user_data):
        if self._is_cloud_uid(user_data.get('uid')):
            return user_data['uid']

        if not user_data.get('gender'):
            user_data['gender'] = 'unknown'

        url = self._construct_url(self.method_templates['account_edit'])
        try:
            json_response = self._make_post_request(url, user_data)
        except exceptions.AccountDisabled:
            # Чтобы отловить проблему приводящую к 422 ошибке при блокировке
            # аккаунта, добавим немного логгирования. Сама проблема описана тут:
            # https://st.yandex-team.ru/TOOLSB-288
            with log.fields(user_data=user_data):
                log.warning('Unable to edit disabled account')
            raise
        return json_response.get('uid')

    def account_delete(self, uid):
        if self._is_cloud_uid(uid):
            return True

        if is_outer_uid(uid):
            with log.fields(uid=uid):
                log.error('Trying to delete outer user, action denied')
            return False

        url = self._construct_url(self.method_templates['account_delete'].format(uid=uid))
        response = self._make_delete_request(url)
        return response.get('status') == 'ok'

    def maillist_add(self, domain, login):
        data = {
            'login': login_to_punycode(login),
            'domain': to_punycode(domain),
            'is_maillist': 1,
            'no_password': 1,
        }

        url = self._construct_url(self.method_templates['account_add'])
        json_response = self._make_post_request(url, data)
        return json_response.get('uid')

    def maillist_delete(self, uid):
        return self.account_delete(uid)

    def set_organization_name(self, domain_id, orgname):
        data = {
            'organization_name': orgname,
        }
        return self.domain_edit(domain_id, data)

    def set_master_domain(self, old_master_id, new_master_id):
        """Устанавливает master домен.
        """
        url = self._construct_url(self.method_templates['change_master_domain'] % (old_master_id, new_master_id))
        response = self._make_post_request(url, None)
        changed = response.get('status') == 'ok'
        return changed

    def del_organization_name(self, domain_id):
        data = {
            'organization_name': '',
        }
        return self.domain_edit(domain_id, data)

    def change_password(self, uid, new_password, force_next_login_password_change=False):
        """
            Смена пароля пользователя
            Описание ручки: https://wiki.yandex-team.ru/passport/api/bundle/account/flushpdd/
            Ошибки: https://wiki.yandex-team.ru/passport/api/bundle/auth/#opisanievoobshhevsexvozmozhnyxoshibok
            Возвращает boolean-статус успешности смены пароля и список ошибок из password_errors при наличии
        """
        if self._is_cloud_uid(uid):
            return True

        url = self._construct_url(self.method_templates['change_password_submit'] % uid)
        response = self._make_post_request(url, {}, log_data=False)

        if response.get('track_id', False):
            # если track_id нет в ответе, значит получили account_not_found и считаем, что пароль поменялся
            track_id = response['track_id']
            data = {
                'password': new_password,
                'track_id': track_id,
            }
            if force_next_login_password_change:
                data['force_password_change'] = 1
            url = self._construct_url(self.method_templates['change_password_commit'] % uid)
            response = self._make_post_request(url, data, log_data=False)
        return response['status'] == 'ok'

    def alias_add(self, uid, alias):
        """
            Добавляет алиас пользователю.
        """
        if self._is_cloud_uid(uid):
            return True

        if not is_login_correct(alias):
            raise exceptions.LoginProhibitedsymbols()
        data = {}
        url_template = self.method_templates['add_alias'].format(uid=uid, alias=alias)
        url = self._construct_url(url_template)
        response = self._make_post_request(url, data)
        return response['status'] == 'ok'

    def _is_cloud_uid(self, uid):
        with log.fields(uid=uid):
            if is_cloud_uid(uid):
                log.warning('Trying to modify cloud user {}, do nothing'.format(uid))
                return True

    def alias_delete(self, uid, alias):
        """
            Удаление алиаса пользователя.
            https://wiki.yandex-team.ru/passport/api/bundle/mdapi/#udaleniealiasapolzovatelja
        """
        if self._is_cloud_uid(uid):
            return True

        url_template = self.method_templates['delete_alias'].format(uid=uid, alias=alias)
        url = self._construct_url(url_template)
        try:
            response = self._make_delete_request(url)
        except exceptions.AliasNotFound:
            return True
        return response['status'] == 'ok'

    def change_avatar(self, uid, file_img=None, url=None):
        """
            Смена аватарки пользователя
            https://wiki.yandex-team.ru/passport/api/bundle/changeavatar/#zagruzkaavatarkipouidsessionnojjkukeilitokenubeztreka
            uid - uid пользователя
            file_img - @werkzeug.datastructures.FileStorage - файл из request
            или
            url - url аватарки
            именно этот метод: по https
        """
        if self._is_cloud_uid(uid):
            return True

        data = {
            'default': True,
            'uid': uid,
        }
        files = None
        if file_img:
            # example: http://docs.python-requests.org/en/master/user/quickstart/#post-a-multipart-encoded-file
            files = {
                'file': (file_img.filename, file_img.stream, file_img.mimetype)
            }
        elif url:
            data['url'] = url

        url = url_join(self.passport_url,
                       self.method_templates['change_avatar'],
                       query_params=self.default_params)
        response = self._make_post_request(
            url,
            data=data,
            headers=self.headers,
            files=files
        )
        return response['status'] == 'ok'

    def block_user(self, uid):
        return self._change_is_enabled(uid, False)

    def unblock_user(self, uid):
        return self._change_is_enabled(uid, True)

    def set_admin_option(self, uid):
        """
        Проставляем признак админа организации
        """

        has_admin_option = self.get_attribute(uid, '154')
        if has_admin_option:
            return True
        return self._change_is_admin(uid, change_to=True)

    def reset_admin_option(self, uid):
        """
        Снимаем признак админа организации
        """

        if is_yandex_team_uid(uid) or self._is_cloud_uid(uid):
            # Для тимной организации не делаем ничего
            # иначе будет ошибка описанная тут:
            # https://st.yandex-team.ru/TOOLSB-200

            # так же ничего не делаем для облачных пользователей
            return None

        # Снимать атрибут нужно только в том случае, если он есть.
        has_admin_option = self.get_attribute(uid, '154')
        if has_admin_option:
            return self._change_is_admin(uid, change_to=False)

    def set_pdd_admin(self, uid):
        if self._is_cloud_uid(uid):
            return True

        has_admin_option = self.get_attribute(uid, '7')
        if has_admin_option:
            return True

        handle = self.method_templates['subscription'].format(
            uid=uid,
            service_slug='pddadmin',
        )
        data = None
        url = self._construct_url(handle)
        return self._make_post_request(url, data)

    def unset_pdd_admin(self, uid):
        if self._is_cloud_uid(uid):
            return True
        has_admin_option = self.get_attribute(uid, '7')
        if not has_admin_option:
            return True

        handle = self.method_templates['subscription'].format(
            uid=uid,
            service_slug='pddadmin',
        )
        data = None
        url = self._construct_url(handle)
        return self._make_delete_request(url, data)

    def _change_is_admin(self, uid, change_to):
        # Данное действие ставит или снимает блокирующий сид на учётке пользователя
        # Подробнее: https://st.yandex-team.ru/DIR-3020
        if self._is_cloud_uid(uid):
            return True

        data = {
            'is_connect_admin': change_to,
        }
        url = self._construct_url(self.method_templates['change_options'] % uid)
        return self._make_post_request(url, data)

    def set_organization_ids(self, uid, org_ids):
        if self._is_cloud_uid(uid):
            return uid

        data = {
            'external_organization_ids': ','.join(map(str, org_ids)),
        }

        url = self._construct_url(
            self.method_templates['change_options'] % uid,
            # directory-portal - это специальный консьюмер, заведённый
            # исключительно под задачу простановки organization_ids.
            # От консьюмера directory он отличается тем, что может
            # изменять настройки не только ПДД учётки, но и любой другой.
            # Больше нигде directory-portal использовать нельзя.
            query_params={'consumer': 'directory-portal'},
        )
        json_response = self._make_post_request(url, data)
        return json_response.get('uid')

    def _change_is_enabled(self, uid, change_to):
        """
        Вернуть True если операция прошла успешно.
        """
        if is_outer_uid(uid):
            log.error('Trying to change outer user "{0}", action denied'.format(uid))
            return False

        if self._is_cloud_uid(uid):
            return True

        data = {
            'is_enabled': change_to,
        }
        url = self._construct_url(self.method_templates['block_user'] % uid)
        response = self._make_post_request(url, data)
        return response.get('status') == 'ok'

    def accept_eula(self, uid):
        """Этот метод принимает за пользователя пользовательское соглашение.

        Внимание! Использовать его нужно только с роботными аккаунтами.
                  Не надо принимать соглашение за обычных пользователей.

        Принятие пользовательского соглашения для роботов пришлось сделать
        потому, что в какой-то момент в паспорте решили, что не надо
        отдавать OAuth токен для аккаунтов, у которых не принято пользователькое
        соглашение. И после этого сломалась работа поиска по Коннекту, так
        как они запрашивают OAuth токен для своего робота и ходят с ним по другим
        сервисам.

        https://st.yandex-team.ru/DIR-3489
        """
        if self._is_cloud_uid(uid):
            return True

        handle = self.method_templates['subscription'].format(
            uid=uid,
            service_slug='pddeula',
        )
        data = None
        url = self._construct_url(handle)
        self._make_post_request(url, data)

    def domain_add(self, domain_name, admin_id):
        data = {
            'domain': domain_name,
            'admin_uid': admin_id,
        }
        url = self._construct_url(self.method_templates['domain_add'])
        try:
            self._make_post_request(url, data)
        except exceptions.DomainAlreadyExists:
            # проверим, что домен принадлежит тому же админу и он мастер
            info = get_domain_info_from_blackbox(domain_name)
            if admin_id == info['admin_id'] and info['master_domain'] is None:
                log.info('Domain already exists in Passport')
            else:
                raise

    def domain_alias_add(self, domain_id, alias):
        data = {'alias': alias}
        url_template = self.method_templates['domain_alias_add'].format(domain_id=domain_id)
        url = self._construct_url(url_template)
        try:
            self._make_post_request(url, data)
        except exceptions.DomainAliasExists:
            # проверим, что пытаемся добавить алиас к тому же домену
            master_domain = get_domain_info_from_blackbox(alias)['master_domain']
            if domain_id == get_domain_info_from_blackbox(master_domain)['domain_id']:
                log.info('Domain alias already exists in Passport')
            else:
                raise

    def domain_alias_delete(self, domain_id, alias_id):
        url_template = self.method_templates['domain_alias_delete'].format(domain_id=domain_id, alias_id=alias_id)
        url = self._construct_url(url_template)
        try:
            return self._make_delete_request(url)
        except exceptions.DomainAliasNotFound:
            return {'status': 'ok'}

    def domain_edit(self, domain_id, data):
        url = self._construct_url(self.method_templates['domain_edit'] % domain_id)
        response = self._make_post_request(url, data)
        return response.get('status') == 'ok'

    def domain_delete(self, domain_id):
        url_template = self.method_templates['domain_delete'].format(domain_id=domain_id)
        url = self._construct_url(url_template)
        return self._make_delete_request(url)

    def create_track(self):
        url = self._construct_url(self.method_templates['create_track'])
        return self._make_post_request(url, None)['id']

    def validate_login(self, login):
        track_id = self.create_track()
        url = self._construct_url(self.method_templates['validate_login'])
        data = {
            'login': login_to_punycode(login),
            'track_id': track_id,
            'is_pdd': 1,
        }
        return self._make_post_request(url, data)

    def validate_password(self, password):
        track_id = self.create_track()
        url = self._construct_url(self.method_templates['validate_password'])
        data = {
            'password': password,
            'track_id': track_id,
        }
        return self._make_post_request(url, data, log_data=False)

    def validate_natural_domain(self, domain):
        url = self._construct_url(self.method_templates['validate_natural_domain'])
        data = {'domain': to_punycode(domain)}
        return self._make_post_request(url, data)

    def validate_connect_domain(self, domain):
        url = self._construct_url(self.method_templates['validate_connect_domain'])
        data = {'domain': to_punycode(domain)}
        return self._make_post_request(url, data)

    def validate_display_name(self, display_name):
        url = self._construct_url(self.method_templates['validate_display_name'])
        data = {'display_name': display_name}
        response = self._make_post_request(url, data)
        return response.get('status') == 'ok'

    def validate_firstname(self, firstname):
        url = self._construct_url(self.method_templates['validate_firstname'])
        data = {'firstname': firstname}
        response = self._make_post_request(url, data)
        return response.get('status') == 'ok'

    def validate_lastname(self, lastname):
        url = self._construct_url(self.method_templates['validate_lastname'])
        data = {'lastname': lastname}
        response = self._make_post_request(url, data)
        return response.get('status') == 'ok'

    @exceptions.account_not_found_handler
    def _make_post_request(self, url, data, headers=None, log_data=True, **kwargs):
        """Этот метод делает запрос в Паспорт, а в случае если тот вернул
        не 200 код ошибки, кидает urllib2.HTTPError.

        А если был 200 ответ, то возвращает JSON с результатом.
        """
        if headers is None:
            headers = self.headers

        try:
            response = http_client.request('post', url, data=data, headers=headers, **kwargs)
        except (requests.exceptions.ConnectionError, requests.exceptions.Timeout):
            raise exceptions.PassportUnavailable()

        if not log_data:
            data = '[REMOVED]'

        try:
            json_response = response.json()
        except:
            # Если JSON распарсить не получается, то залоггируем ответ как есть
            self._write_log(url, data, response.status_code, response.text, trace=True)
            raise

        self._write_log(url, data, response.status_code, json_response)
        response.raise_for_status()

        raise_exception_if_errors(json_response)

        return json_response

    @exceptions.account_not_found_handler
    def _make_delete_request(self, url, headers=None, **kwargs):
        headers = headers if headers is not None else self.headers

        try:
            response = http_client.request('delete', url, headers=headers, **kwargs)
        except (requests.exceptions.ConnectionError, requests.exceptions.Timeout):
            raise exceptions.PassportUnavailable

        try:
            json_response = response.json()
        except:
            # Если JSON распарсить не получается, то залоггируем ответ как есть
            self._write_log(url, '[REMOVED]', response.status_code, response.text, trace=True)
            raise

        self._write_log(url, '[REMOVED]', response.status_code, json_response)

        raise_exception_if_errors(json_response)
        return json_response

    def _change_is_maillist(self, uid, is_maillist):
        """Вспомогательная функция для изменения флага, указывающего на то,
           является ли аккаунт на яндексе обычным ящиком или рассылкой.
        """
        if self._is_cloud_uid(uid):
            return True
        data = {
            'is_maillist': 'yes' if is_maillist else 'no',
        }
        url = self._construct_url(self.method_templates['change_options'] % uid)
        response = self._make_post_request(url, data)
        # Ручка должна вернуть {u'status': 'ok'}, ошибки тут проверять не надо,
        # потому что это уже сделал метод _make_post_request.
        return response

    def make_maillist(self, uid):
        """Сделать ящик рассылкой"""
        return self._change_is_maillist(uid, True)

    def make_email(self, uid):
        """Сделать рассылку ящиком."""
        return self._change_is_maillist(uid, False)


    def get_attribute(self, uid, attribute, user_ip=None, default=None):
        """Вспомогательная функция для получения атрибута из blackbox."""
        if user_ip is None:
            user_ip = get_localhost_ip_address()

        response = app.blackbox_instance.userinfo(
            uid=uid,
            userip=user_ip,
            attributes=attribute,
        )
        return response.get('attributes', {}).get(attribute, default)

    def block_domain(self, domain_id):
        return self.domain_edit(domain_id, {
            'enabled': False,
        })

    def unblock_domain(self, domain_id):
        return self.domain_edit(domain_id, {
            'enabled': True,
        })

    def global_logout_domain(self, domain_id):
        return self.domain_edit(domain_id, {
            'glogouted': True,
        })


class TeamPassportApiClient(PassportApiClient):
    def __init__(self):
        super(TeamPassportApiClient, self).__init__()
        self.passport_url = app.config['TEAM_PASSPORT_API']
        self.method_templates.update({
            'yambot_add': '/1/bundle/account/register/yambot/'
        })

    def yambot_add(self):
        '''
        Метод для заведения ботов Ямб в тимном паспорте
        https://wiki.yandex-team.ru/passport/api/bundle/registration/#registracijabotadljajamba
        '''
        url = self._construct_url(self.method_templates['yambot_add'])
        response = self._make_post_request(url, {})
        return response.get('uid')


def build_passport_exception_response(exception):
    return json_error(
        exception.http_code,
        exception.code,
        exception.message,
    )


def to_punycode(text):
    try:
        return force_text(text).encode('idna')
    except UnicodeError:
        raise exceptions.PassportException


def is_login_correct(login):
    return LOGIN_VALIDATE_RE.match(login) is not None
