# -*- coding: utf-8 -*-
from functools import wraps
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


class PassportException(Exception):
    code = 'unknown'
    message = 'Passport Unexpected Error: {error}'
    http_code = 503
    log_level = 'ERROR'

    def __init__(self, error=None):
        super(Exception, self).__init__()
        if error:
            self.error_code = error
        if '{error}' in self.message:
            self.message = self.message.format(error=error)

    def __str__(self):
        return self.message


class PassportWarning(PassportException):
    log_level = 'WARNING'
    http_code = 422


class PassportUnavailable(PassportException):
    message = 'Passport is not available'
    code = 'passport.not_available'
    http_code = 503


# password
class PasswordEmpty(PassportWarning):
    message = 'Password is empty'
    code = 'password.empty'


class PasswordProhibitedsymbols(PassportWarning):
    message = 'Prohibited symbols in password'
    code = 'password.prohibitedsymbols'


class PasswordWeak(PassportWarning):
    message = 'Password is too weak'
    code = 'password.weak'


class PasswordShort(PassportWarning):
    message = 'Password is too short'
    code = 'password.short'


class PasswordLong(PassportWarning):
    message = 'Password is too long'
    code = 'password.long'


class PasswordLikelogin(PassportWarning):
    message = 'Password is equal to login'
    code = 'password.likelogin'

class YandexMailToOrgWithDomain(PassportWarning):
    message = "Can't add yandex mail to org with that domain into occupied login"
    code = 'email.occupied'


class PasswordLikephonenumber(PassportWarning):
    message = 'Password is equal to phone number'
    code = 'password.likephonenumber'


class PasswordEqualsPrevious(PassportWarning):
    message = 'Password equals previous'
    code = 'password.equals_previous'


class PasswordFoundInHistory(PassportWarning):
    message = 'Password found in history'
    code = 'password.found_in_history'


class BirthdayInvalid(PassportWarning):
    message = 'Birthday is invalid'
    code = 'invalid_birthday'


# login
# NB: new
class LoginEmpty(PassportWarning):
    message = 'Login is empty'
    code = 'login.empty'


# NB: new
class LoginProhibitedsymbols(PassportWarning):
    message = 'Prohibited symbols in login'
    code = 'login.prohibitedsymbols'


# NB: new
class LoginLong(PassportWarning):
    message = 'Login is too long'
    code = 'login.long'


# NB: new
class LoginNotavailable(PassportWarning):
    message = 'Passport account already exist'
    code = 'login.notavailable'
    http_code = 409


# TODO: удалить после закрытия https://st.yandex-team.ru/PASSP-20119
class LoginNotAvailable(PassportWarning):
    message = 'Passport account already exist'
    code = 'login.notavailable'
    http_code = 409

# domain
# NB: new
class DomainNotFound(PassportException):
    message = 'Domain not found in passport'
    # code = 'domain.not_found'
    code = 'domain_not_register_in_passport'
    http_code = 404


class DomainInvalidType(PassportException):
    message = 'Empty or not RFC domain name'
    # code = 'domain.invalid_type'
    code = 'bad_domain'
    http_code = 422

class DomainInvalid(PassportException):
    message = 'Empty or not RFC domain name'
    # code = 'domain.invalid'
    code = 'bad_domain'
    http_code = 422

class DomainYandexTeam(PassportException):
    message = '"yandex-team" domain not supported by external passport'
    code = 'domain.yandex_team'
    http_code = 422


# account
class AccountDisabled(PassportException):
    message = 'Account is disabled'
    code = 'account.disabled'
    http_code = 422


class AccountNotFound(PassportException):
    message = 'Account not found'
    code = 'not_found'
    http_code = 422


# NB: new
class AccountHasBlockingSids(PassportException):
    message = 'Account has blocking SIDs'
    code = 'has_blocking_sids'
    http_code = 422


# form
# ToDo: art@ неправильный, на мой взгляд, маппинг.
# file.error вместо change_avatar.error было бы правильно
class FormInvalid(PassportException):
    message = 'Validation error'
    code = 'change_avatar.error'
    http_code = 422


# file
# ToDo: art@ неправильный, на мой взгляд, маппинг.
# file.invalid вместо change_avatar.file_invalid было бы правильно
class FileInvalid(PassportException):
    message = 'Invalid file for avatar'
    code = 'change_avatar.file_invalid'
    http_code = 422


# organization
class OrganizationAccountLimit(PassportException):
    message = 'Organization account limit was exceeded'
    code = 'account_has_limit'
    http_code = 409


# change_avatar
class ChangeAvatarInvalidImageSize(PassportException):
    message = 'Invalid image size for avatar'
    code = 'change_avatar.invalid_image_size'
    http_code = 422


class ChangeAvatarInvalidFileSize(PassportException):
    message = 'Invalid file size for avatar'
    code = 'change_avatar.invalid_file_size'
    http_code = 422


class ChangeAvatarInvalidUrl(PassportException):
    message = 'Invalid url for avatar'
    code = 'change_avatar.invalid_url'
    http_code = 422


# alias
class AliasExists(PassportWarning):
    message = 'Alias exists'
    code = 'alias_exists'


class AliasNotFound(PassportException):
    message = 'Alias not found in Passport'
    code = 'alias_not_found'
    http_code = 422


class DomainAlreadyExists(PassportException):
    message = 'Domain already exists'
    code = 'domain_already_exists'
    http_code = 422


class DomainAliasExists(PassportException):
    message = 'Domain alias already exists'
    code = 'domain_alias_already_exists'
    http_code = 422


class DomainAliasNotFound(PassportException):
    message = 'Domain alias not found'
    code = 'domain_alias_not_found'
    http_code = 422


class InvalidTimezone(PassportException):
    message = 'Unknown timezone'
    code = 'timezone.invalid'
    http_code = 422


class InvalidLanguage(PassportException):
    message = 'Unknown language'
    code = 'language.invalid'
    http_code = 422


class FirstnameInvalid(PassportException):
    message = 'Invalid first name'
    code = 'firstname.invalid'
    http_code = 422


class LastnameInvalid(PassportException):
    message = 'Invalid last name'
    code = 'lastname.invalid'
    http_code = 422


class DisplayNameInvalid(PassportException):
    message = 'Invalid display name'
    code = 'display_name.invalid'
    http_code = 422


def account_not_found_handler(func):
    # декоратор, чтобы логгировать ошибки паспорта AccountNotFound и продолжать работу
    @wraps(func)
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except AccountNotFound:
            kwargs.update(dict(list(zip(list(map(str, list(range(len(args))))), args))))  # преобразуем args в словарь
            with log.fields(**kwargs):
                log.warning('Account not found in Passport')
            return {'status': 'ok'}
    return wrapper
