# -*- coding: utf-8 -*-
import contextlib
import re

from flask import Blueprint

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.admin.views.organizations import \
    AdminOrganizationChangeSubscriptionPlanView, \
    AdminOrganizationServicesEnableView, AdminOrganizationServicesDisableView, OrganizationBlockView, \
    OrganizationUnblockView

from intranet.yandex_directory.src.yandex_directory.admin.views.resource_history import ResourceHistoryView
from intranet.yandex_directory.src.yandex_directory.admin.views.tasks import AdminTasksDependeciesListView
from intranet.yandex_directory.src.yandex_directory.core.views.lego.view import OrganizationsByYandexSessionView, OrganizationsByUidsView
from intranet.yandex_directory.src.yandex_directory.directory_logging.flask_logging_extension import (
    BaseLoggingExtension,
)
from intranet.yandex_directory.src.yandex_directory.connect_services.alice_b2b import setup_alice_b2b_client
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    BlackboxWithTimings,
    create_requests_session,
    BlackboxRoutingProxy,
)

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_error_not_found,
)
from intranet.yandex_directory.src.yandex_directory.common.web import DefaultErrorHandlingExtension
from intranet.yandex_directory.src.yandex_directory.connect_services.direct.client import setup_direct_client
from intranet.yandex_directory.src.yandex_directory.connect_services.metrika.client import setup_metrika_client
from intranet.yandex_directory.src.yandex_directory.connect_services.domenator import setup_domenator_client
from intranet.yandex_directory.src.yandex_directory.common.caching import setup_caching

from intranet.yandex_directory.src.yandex_directory.core.views.api_versioning import build_cached_view_methods_map
from intranet.yandex_directory.src.yandex_directory.admin.views.users import UserContactsView
from intranet.yandex_directory.src.yandex_directory.core.views.domains.add_owned_domains_view import AddOwnedDomainsView
from intranet.yandex_directory.src.yandex_directory.core.views.domains.domain_token import GetDomainAndAdminUidView, DomainTokenView
from intranet.yandex_directory.src.yandex_directory.core.views.groups import GroupMembersBulkAdminUpdateView
from intranet.yandex_directory.src.yandex_directory.common.exitstack import ExitStackExtension
from intranet.yandex_directory.src.yandex_directory.core.views.service.roles_list_view import RolesListView
from intranet.yandex_directory.src.yandex_directory.main.config import MainComponentsConfig
from intranet.yandex_directory.src.yandex_directory.meta.config import MetaComponentsConfig
from intranet.yandex_directory.src.yandex_directory.limits.config import OrganizationLimitsComponentsConfig
from intranet.yandex_directory.src.yandex_directory.connect_services.yandexsprav import setup_yandexsprav_client
from intranet.yandex_directory.src.yandex_directory.connect_services.partner_disk import setup_partner_disk_client
from intranet.yandex_directory.src.yandex_directory.core.views.organization.view import OrganizationDebtView, \
    OrganizationMetaView
from intranet.yandex_directory.src.yandex_directory.admin.views.licenses import AdminTrackerLicensesView
from intranet.yandex_directory.src.yandex_directory.core.views.domenator.views import DomainOccupiedView
from intranet.yandex_directory.src.yandex_directory.auth.middlewares import LogMiddleware
from intranet.yandex_directory.src.yandex_directory.core.cloud.claims import CloudClaimServiceClient
from intranet.yandex_directory.src.yandex_directory.sso.views import SsoSettingsView, SsoActionsView


class VersionDispatcher(object):
    """
    Получает из URL запрошенную версию API, если она там есть.
    Если нет, считаем что запрошена первая версия (v1).
    Заменяет запрошенный путь на путь без версии (/v1/users/ -> /users/).
    Версия API будет находиться в request.api_version, её положит туда миддлвара APIVersionMiddleware
    из окружения wsgi приложения.
    """

    def __init__(self, app):
        self.app = app
        self.find_version_regexp = re.compile(r'^(/?v(?P<version>\d+))?(?P<path>/?.*)$')

    def __call__(self, environ, start_response):
        environ['API_VERSION'] = self._get_version(environ)
        return self.app(environ, start_response)

    def _get_version(self, environ):
        matched = self.find_version_regexp.match(environ.get('PATH_INFO', ''))
        if matched:
            return matched.groupdict()['version'] or '1'
        else:
            return '1'


def server_port_fixer(next_wsgi_app):
    """В продакшене, на IPv6 машинке почему-то в environ['SERVER_PORT']
       попадает значение ':]:80'.

       Из-за этого, любая ручка пятисотит, если не передан заголовок Host,
       и не работает healthcheck Qloud.

       Эта небольшая WSGI миддльварь чинит данную проблему.

       https://st.yandex-team.ru/DIR-4228
    """

    def fix_server_port(environ, start_response):
        if environ['SERVER_PORT'].startswith(':'):
            environ['SERVER_PORT'] = environ['SERVER_PORT'].rsplit(':', 1)[-1]
        return next_wsgi_app(environ, start_response)

    return fix_server_port


def view_methods(view_class):
    """Это генератор, позволяющий итерироваться по HTTP методам класса View.
    """
    for http_method in view_class.methods:
        http_method = http_method.lower()

        for method_name in dir(view_class):
            if method_name.startswith(http_method):
                method = getattr(view_class, method_name)
                yield method_name, method


def is_internal_view(cls):
    # Проверим все методы view, реализующие ручки API
    # и если хотя бы один из них internal, то
    # вернём True
    for name, method in view_methods(cls):
        internal = getattr(method, 'internal', None)
        if internal:
            return True


def check_if_view_has_scopes_and_permissions(cls):
    for name, method in view_methods(cls):
        scopes = getattr(method, 'scopes', None)
        if scopes is None:
            # Значит скоупы не указаны и это проблема
            raise AssertionError(
                'Scopes not given for handle {cls}.{method}'.format(
                    cls=cls,
                    method=name,
                )
            )

        permissions = getattr(method, 'permissions', None)
        if permissions is None:
            # Значит права не указаны и это проблема
            raise AssertionError(
                'Permissions not given for handle {module}:{cls}.{method}'.format(
                    module=cls.__module__,
                    cls=cls.__name__,
                    method=name,
                )
            )


@contextlib.contextmanager
def versioned_blueprint(app, bp_name, import_name):
    bp = Blueprint(bp_name, import_name)
    try:
        yield bp
    finally:
        app.register_blueprint(bp, url_defaults=dict(api_version=1))
        app.register_blueprint(bp, url_prefix='/v<int:api_version>')


def initialize_routes(app, internal_api):
    """Добавляет в приложение роуты.
    Раньше у нас этот код исполнялся на верхнем уровне, но
    был вынесен в отдельную функцию для того, чтобы можно было
    запустить его повторно при генерации сваггер-спеки, и включить
    внутренние ручки независимо от того, какая настройка в app.config['INTERNAL']

    На входе ожидает True, если запускается внутреннее API и False если внешнее.
    """

    external_routes = []
    internal_routes = []

    for i, (uri, view) in enumerate(app.ROUTES):
        if isinstance(view, tuple):
            view, defaults = view
        else:
            defaults = {}
        check_if_view_has_scopes_and_permissions(view)
        name = '{}{}'.format(view.__name__, i)
        view_func = getattr(view, 'as_view')(name)

        if is_internal_view(view):
            internal_routes.append((uri, view_func, defaults))
        else:
            external_routes.append((uri, view_func, defaults))

    if 'external' not in app.blueprints:
        with versioned_blueprint(app, 'external', __name__) as bp:
            for uri, view_func, defaults in external_routes:
                bp.add_url_rule(uri, defaults=defaults, view_func=view_func)

    if internal_api and 'internal' not in app.blueprints:
        with versioned_blueprint(app, 'internal', __name__) as bp:
            for uri, view_func, defaults in internal_routes:
                bp.add_url_rule(uri, defaults=defaults, view_func=view_func)

    # Если происходит переинициализация, то нужно заново перестроить
    # кэш вьюшек
    app.cached_view_methods = build_cached_view_methods_map()


def setup_app(app):
    # Защищаем приложение от повторной настройки
    if getattr(app, '__setup_complete__', False):
        return app
    else:
        setattr(app, '__setup_complete__', True)

    import sys
    from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import error_log

    def handle_exception(exc_type, exc_value, exc_traceback):
        if issubclass(exc_type, KeyboardInterrupt):
            sys.__excepthook__(exc_type, exc_value, exc_traceback)
            return

        error_log.error("Uncaught exception", exc_info=(exc_type, exc_value, exc_traceback))
    sys.excepthook = handle_exception

    app.url_map.redirect_defaults = False
    app.secret_key = app.config['SECRET_KEY']

    BaseLoggingExtension(app)

    from tvmauth import BlackboxTvmId as BlackboxClientId

    app.yandex_blackbox_instance = BlackboxWithTimings(
        app.config['BLACKBOX']['url'],
        tvm2_client_id=app.config['TVM_CLIENT_ID'],
        tvm2_secret=app.config['TVM_SECRET'],
        blackbox_client=(
            BlackboxClientId.Prod
            if app.config['ENVIRONMENT'] in app.config['PRODUCTION_ENVIRONMENTS']
            else BlackboxClientId.Test
        ),
    )

    app.cloud_blackbox_instance = CloudClaimServiceClient()

    app.blackbox_instance = BlackboxRoutingProxy(
        yandex_bb=app.yandex_blackbox_instance,
        cloud_bb=app.cloud_blackbox_instance,
    )

    app.team_blackbox_instance = BlackboxWithTimings(
        app.config['TEAM_BLACKBOX']['url'],
        tvm2_client_id=app.config['TVM_CLIENT_ID'],
        tvm2_secret=app.config['TVM_SECRET'],
        blackbox_client=(
            BlackboxClientId.ProdYateam
        ),
    )

    from intranet.yandex_directory.src.yandex_directory.common.views.main import (
        IndexView,
        TestNotify,
        PingView,
        SimplePingView,
        ExceptionView,
        StatsView,
    )
    from intranet.yandex_directory.src.yandex_directory.auth.middlewares import (
        AuthMiddleware,
        IncomingRequestStatsMiddleware,
        RequestIDMiddleware,
        DebugSwitcherMiddleware,
        ValidateOutputData,
        DCMiddleware,
        OrganizationRevisionMiddleware,
        APIVersionMiddleware,
        CloseServiceBySmokeTestResultsMiddleware,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.permissions import PermissionListView
    from intranet.yandex_directory.src.yandex_directory.core.views.domains import (
        DomainsListView,
        DomainsSearchView,
        DomainDetailView,
        DomainCheckOwnershipView,
        DomainOwnershipInfoView,
        DomainConnectView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.users import (
        UserListView,
        UserDetailView,
        UserChangeAvatarView,
        UserAliasesListView,
        UserAliasesDetailView,
        UserSettingsListView,
        UserTokenView,
        UsersBulkUpdateView,
        UserByNicknameView,
        UserOuterDeputyAdminListView,
        UserOuterDeputyAdminDetailView,
        UserTypeView,
        OuterAdminMigrationView,
        CloudUserDetailView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.departments import (
        DepartmentListView,
        DepartmentDetailView,
        DepartmentBulkMove,
        DepartmentAliasesListView,
        DepartmentAliasesDetailView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.disk_usage import DiskUsageView
    from intranet.yandex_directory.src.yandex_directory.core.views.actions import ActionListView
    from intranet.yandex_directory.src.yandex_directory.core.views.events import EventListView
    from intranet.yandex_directory.src.yandex_directory.core.views.proxy import ProxyOrganizationsView
    from intranet.yandex_directory.src.yandex_directory.core.views.groups import (
        GroupListView,
        GroupDetailView,
        GroupMembersView,
        GroupMembersBulkUpdateView,
        GroupAliasesListView,
        GroupAliasesDetailView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.resources import (
        ResourceListView,
        ResourceCountView,
        ResourceDetailView,
        ResourceDetailRelationsView,
        ResourceBulkUpdateRelationsView
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.relations import (
        RelationDetailView,
        RelationListView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.organization.view import (
        OrganizationView,
        OrganizationsView,
        OrganizationByOrgIdView,
        OrganizationFeaturesByOrgIdView,
        OrganizationFeaturesChangeView,
        OrganizationByLabelView,
        OrganizationsChangeLogoView,
        OrganizationWithDomainView,
        OrganizationWithoutDomainView,
        OrganizationsChangeOwnerView,
        OrganizationFromRegistrarView,
        OrganizationsDeleteView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.dependencies import (
        DependenciesView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.subscription import (
        OrganizationChangeSubscriptionPlanView,
        OrganizationSubscriptionPricingView,
        OrganizationPayView,
        OrganizationInvoiceView,
        OrganizationSubscriptionInfoView,
        OrganizationChangeServiceLicensesView,
        OrganizationDeleteServiceLicensesView,
        OrganizationCreateContractInfoView,
        OrganizationCalculatePriceServiceLicensesView,
        OrganizationPromocodeActivateView,
        OrganizationRequestLicensesView,
        OrganizationManageRequestedLicensesView,
        OrganizationDownloadContractView,
        OrganizationSubscriptionAdminPersonsView,
        OrganizationTrustPayView,
        OrganizationTrackerDetailLicensesView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.webhooks import (
        WebHookListView,
        WebHookDetailView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.invite import (
        InviteView,
        InviteDetailView,
        InviteUseView,
        InviteEmailsView,
        DepartmentInviteView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views import registrar
    from intranet.yandex_directory.src.yandex_directory.core.views.service import (
        OrganizationServiceActionView,
        OrganizationServiceDetailView,
        OrganizationServiceReadyView,
        ServicesListView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.suggest import (
        SuggestView,
        SuggestLicenseView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.ui import (
        UIHeaderView,
        UIOrganizationProfileInfoView,
        UIOrganizationsView,
        UIServicesView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.mail_migration import MailMigrationView
    from intranet.yandex_directory.src.yandex_directory.swagger import enable_swagger_for
    from intranet.yandex_directory.src.yandex_directory.core.views.who_is import WhoIsView
    from intranet.yandex_directory.src.yandex_directory.sso.views import SsoAllowedView
    from intranet.yandex_directory.src.yandex_directory.core.views.tasks import (
        TaskDetailView,
        TaskListView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.binding_widget.views import (
        BindOrganizationView,
        CheckOrganizationsBindingAvailable,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.check_mail_server import CheckMailServerView
    from intranet.yandex_directory.src.yandex_directory.core.views.antispam import SODomainInfoView
    from intranet.yandex_directory.src.yandex_directory.core.views.cloud.user_cloud_organizations_view import \
        UserCloudOrganizationsView
    from intranet.yandex_directory.src.yandex_directory.core.views.cloud.eventhandler import CloudEventHandleView
    from intranet.yandex_directory.src.yandex_directory.core.monitoring import setup_stats_aggregator
    from intranet.yandex_directory.src.yandex_directory.common.db import (
        setup_database,
        setup_engines,
    )
    from intranet.yandex_directory.src.yandex_directory.common.billing import setup_billing
    from intranet.yandex_directory.src.yandex_directory.common.yt.utils import setup_yt
    from intranet.yandex_directory.src.yandex_directory.core.task_queue.base import setup_task_queue
    from intranet.yandex_directory.src.yandex_directory.core.mailer import setup_mailer
    from intranet.yandex_directory.src.yandex_directory.core.sms import setup_sms
    from intranet.yandex_directory.src.yandex_directory.core.templates import setup_templates
    from intranet.yandex_directory.src.yandex_directory.core.scheduler.cron import setup_cron
    from intranet.yandex_directory.src.yandex_directory.auth.tvm import setup_tvm
    from intranet.yandex_directory.src.yandex_directory.passport import setup_passport
    from intranet.yandex_directory.src.yandex_directory.core.idm import setup_idm
    from intranet.yandex_directory.src.yandex_directory.common.yql import setup_yql
    from intranet.yandex_directory.src.yandex_directory.zora import setup_zora
    from intranet.yandex_directory.src.yandex_directory.core.mailer.cloud_notify import setup_cloud_notifier

    setup_engines(app)
    setup_passport(app)
    setup_stats_aggregator(app)
    setup_database(app)
    setup_billing(app)
    setup_yt(app)
    setup_task_queue(app)
    setup_mailer(app)
    setup_sms(app)
    setup_templates(app)
    setup_cron(app)
    setup_tvm(app)
    setup_idm(app)
    setup_yql(app)
    setup_cloud_notifier(app)
    setup_caching(app)

    LogMiddleware(app)
    DefaultErrorHandlingExtension(app)
    MetaComponentsConfig(app)
    MainComponentsConfig(app)
    OrganizationLimitsComponentsConfig(app)

    # сохраняем полученную в VersionDispatcher запрошенную версию API в request.api_version
    APIVersionMiddleware(app)

    # Пишет в голован времена обработки исходящих запросов
    #
    # ВНИМАНИЕ: Эта миддльварь должна идти первой, иначе получается
    #           что время замеряется неправильно.
    #
    IncomingRequestStatsMiddleware(app)

    # закрывает все ручки сервиса кроме /ping/ если
    # по результатам smoke тестов недоступны жизненно важные системы
    # несколько проверок подряд
    CloseServiceBySmokeTestResultsMiddleware(app)

    # обрабатывает заголовок x-debug и включает отладку
    # когда он передан, в лог будут писаться времена обработки
    # блоков, обернутых в конткстный менеджер LogWorkTime
    # в лог пишутся записи вида: metric=%s value=%.2f
    DebugSwitcherMiddleware(app)

    # для запроса генерируется уникальный id
    # он добавляется в виде поля в записи лога
    # и выводится в заголовке X-Request-ID
    RequestIDMiddleware(app)

    # Поддержка ExitStack для request/app_context
    # В него удобно класть контекстные мэнеджеры
    # см. https://contextlib2.readthedocs.io/en/stable/#contextlib2.ExitStack
    ExitStackExtension(app)

    # В ответы с кодом < 500 добавляем заголовок X-Revision с ревизией организации
    OrganizationRevisionMiddleware(app)

    # добавляет заголовок X-DC с названием датацентра
    # (только для запущенных в qloud инстансов)
    DCMiddleware(app)

    # эта миддльварь проверяет что за пользователь или сервис пришел, и в
    # зависимости от того, какие параметры нужны view, может кинуть
    # исключение AuthorizationError
    # После этой миддлевари, нам становятся известны:
    # g.user
    # g.service
    # g.org_id
    AuthMiddleware(app)

    # в тестинге, валидирует выходные данные по схеме
    ValidateOutputData(app)

    app.ROUTES = [
        ('/', IndexView),
        ('/testnotify/', TestNotify),
        ('/ping/', PingView),
        ('/simple-ping/', SimplePingView),
        ('/exception/', ExceptionView),
        ('/monitoring/', StatsView),
        ('/permissions/', PermissionListView),
        ('/users/', UserListView),
        ('/users/<int:user_id>/', UserDetailView),
        ('/users/cloud/<cloud_uid>/', CloudUserDetailView),
        ('/users/<int:user_id>/aliases/', UserAliasesListView),
        ('/users/<int:user_id>/token/', UserTokenView),
        ('/users/<int:user_id>/aliases/<alias>/', UserAliasesDetailView),
        ('/users/<int:user_id>/change-avatar/', UserChangeAvatarView),
        ('/users/type/', UserTypeView),
        ('/users/bulk-update/', UsersBulkUpdateView),
        ('/users/nickname/<nickname>/', UserByNicknameView),
        ('/users/deputy/', UserOuterDeputyAdminListView),
        ('/users/deputy/<nickname>/', UserOuterDeputyAdminDetailView),
        ('/settings/', UserSettingsListView),
        ('/departments/', DepartmentListView),
        ('/departments/<int:department_id>/', DepartmentDetailView),
        ('/departments/<int:department_id>/aliases/', DepartmentAliasesListView),
        ('/departments/<int:department_id>/aliases/<alias>/', DepartmentAliasesDetailView),
        ('/departments/<int:department_id>/invite/', DepartmentInviteView),
        ('/domains/', DomainsListView),
        ('/domains/search/', DomainsSearchView),
        ('/domains/<domain_name>/', DomainDetailView),
        ('/domains/<domain_name>/check-ownership/', DomainCheckOwnershipView),
        ('/domains/<domain_name>/ownership-info/', DomainOwnershipInfoView),
        ('/domains/connect/', DomainConnectView),
        ('/groups/', GroupListView),
        ('/groups/<int:group_id>/', GroupDetailView),
        ('/groups/<int:group_id>/members/', GroupMembersView),
        ('/groups/<int:group_id>/members/bulk-update/', GroupMembersBulkUpdateView),
        ('/groups/<int:group_id>/admins/bulk-update/', GroupMembersBulkAdminUpdateView),
        ('/groups/<int:group_id>/aliases/', GroupAliasesListView),
        ('/groups/<int:group_id>/aliases/<alias>/', GroupAliasesDetailView),
        ('/resources/', ResourceListView),
        ('/resources/count/', ResourceCountView),
        ('/resources/<resource_id>/', ResourceDetailView),
        ('/resources/<resource_id>/relations/', ResourceDetailRelationsView),
        ('/resources/<resource_id>/relations/bulk-update/', ResourceBulkUpdateRelationsView),
        ('/relations/', RelationListView),
        ('/relations/<relation_id>/', (RelationDetailView, {'service_slug': None})),
        ('/relations/<service_slug>/<relation_id>/', RelationDetailView),
        ('/organization/', OrganizationView),
        ('/organization/with-domain/', OrganizationWithDomainView),
        ('/organization/from-registrar/<int:registrar_id>/', OrganizationFromRegistrarView),
        ('/organization/without-domain/', OrganizationWithoutDomainView),
        ('/organizations/', OrganizationsView),
        ('/organizations/<int:org_id>/', OrganizationByOrgIdView),
        ('/organizations/sso/settings/', SsoSettingsView),
        ('/organizations/sso/<action>/', SsoActionsView),
        ('/organizations/sso/allowed/', SsoAllowedView),
        ('/organizations/<int:org_id>/features/', OrganizationFeaturesByOrgIdView),
        ('/organizations/<int:org_id>/meta/<key>/', OrganizationMetaView),
        ('/organizations/<int:org_id>/features/<feature_slug>/<action>/', OrganizationFeaturesChangeView),
        ('/organizations/<int:org_id>/change-logo/', OrganizationsChangeLogoView),
        ('/organizations/<int:org_id>/change-owner/', OrganizationsChangeOwnerView),
        ('/organizations/<int:org_id>/delete/', OrganizationsDeleteView),
        ('/organizations/<int:org_id>/debt/', OrganizationDebtView),
        ('/subscription/create-contract-info/', OrganizationCreateContractInfoView),
        ('/subscription/change/', OrganizationChangeSubscriptionPlanView),
        ('/subscription/pricing/', OrganizationSubscriptionPricingView),
        ('/subscription/persons/', OrganizationSubscriptionAdminPersonsView),
        ('/subscription/trust/pay/', OrganizationTrustPayView),
        ('/subscription/pay/', OrganizationPayView),
        ('/subscription/invoice/', OrganizationInvoiceView),
        ('/subscription/services/<service_slug>/licenses/', OrganizationChangeServiceLicensesView),
        ('/subscription/services/<service_slug>/licenses/<obj_type>/<obj_id>/', OrganizationDeleteServiceLicensesView),
        ('/subscription/services/<service_slug>/licenses/calculate-price/',
         OrganizationCalculatePriceServiceLicensesView),
        ('/subscription/services/<service_slug>/licenses/request/', OrganizationRequestLicensesView),
        ('/subscription/services/<service_slug>/licenses/request/<action>/', OrganizationManageRequestedLicensesView),
        ('/subscription/services/tracker/licenses/detail', OrganizationTrackerDetailLicensesView),
        ('/subscription/promocodes/activate/', OrganizationPromocodeActivateView),
        ('/subscription/contract/download/', OrganizationDownloadContractView),
        ('/subscription/', OrganizationSubscriptionInfoView),
        ('/organization/<label>/', OrganizationByLabelView),
        ('/disk-usage/', DiskUsageView),
        ('/suggest/', SuggestView),
        ('/suggest-license/<service_slug>/', SuggestLicenseView),
        ('/actions/', ActionListView),
        ('/events/', EventListView),
        ('/invites/', InviteView),
        ('/invites/emails/', InviteEmailsView),
        ('/invites/<code>/', InviteDetailView),
        ('/invites/<code>/use/', InviteUseView),
        ('/ui/header/', UIHeaderView),
        ('/ui/org-profile/', UIOrganizationProfileInfoView),
        ('/ui/organizations/', UIOrganizationsView),
        ('/ui/services/', UIServicesView),
        ('/services/', ServicesListView),
        # Ручка /ready/ добавлена отдельно, так как она не требует специальных скоупов
        # а вместо этого проверяет право сервиса изменять свой собственный статус.
        ('/services/<service_slug>/ready/', OrganizationServiceReadyView),
        ('/services/<service_slug>/<action>/', OrganizationServiceActionView),
        ('/services/<service_slug>/', OrganizationServiceDetailView),
        ('/services/<service_slug>/roles/', RolesListView),
        ('/bind/', BindOrganizationView),
        ('/bind/organizations/', CheckOrganizationsBindingAvailable),
        ('/dependencies/', DependenciesView),
        ('/webhooks/', WebHookListView),
        ('/webhooks/<int:webhook_id>/', WebHookDetailView),
        ('/who-is/', WhoIsView),

        ('/tasks/', TaskListView),
        ('/tasks/<task_id>/', TaskDetailView),
        ('/move/', DepartmentBulkMove),

        ('/mail-migration/', MailMigrationView),
        ('/mail-migration/check-server/', CheckMailServerView),

        ('/registrar/<id>/', registrar.RegistrarView),
        ('/registrar/token/v2/<token>/', registrar.RegistarV2ByTokenView),
        ('/registrar/uid/v2/<int:uid>/', registrar.RegistarV2ByUidView),
        ('/registrar/<int:registrar_id>/token/', registrar.RegistarV2TokenView),

        ('/domain-token/<token>/<pdd_version>/', GetDomainAndAdminUidView),
        ('/domain-token/<int:uid>/<domain_name>/<pdd_version>', DomainTokenView),

        ('/proxy/organizations/', ProxyOrganizationsView),
        ('/proxy/domains/', AddOwnedDomainsView),

        ('/so/domaininfo/', SODomainInfoView),

        ('/organizations-by-session/', OrganizationsByYandexSessionView),
        ('/organizations-by-uids/', OrganizationsByUidsView),

        ('/admin/migration/', OuterAdminMigrationView),

        ('/domenator/event/domain-occupied/', DomainOccupiedView),

        ('/users/tracker_ids/', UserCloudOrganizationsView),
        ('/cloud/event/', CloudEventHandleView),
    ]

    @app.url_value_preprocessor
    def remove_api_version_from_args(endpoint, values):
        if values:
            values.pop('api_version', None)

    # Добавим все роуты в приложение
    initialize_routes(app, app.config['INTERNAL'])

    # admin
    from intranet.yandex_directory.src.yandex_directory.admin.views.task_names import (
        TaskNamesView,
    )
    from intranet.yandex_directory.src.yandex_directory.admin.views.restore import (
        AdminOrganizationAccessRestoreListView
    )
    from intranet.yandex_directory.src.yandex_directory.admin.views.organizations import (
        AdminOrganizationsListView,
        AdminOrganizationDetailView,
        AdminOrganizationTotalCount,
        AdminOrganizationAdminsListView,
        AdminOrganizationServicesListView,
        AdminOrganizationServicesDetailView,
        AdminOrganizationChangeType,
        AdminOrganizationVerifyDomainView,
        AdminOrganizationsActionsListView,
        AdminOrganizationStaffListView,
        AdminOrganizationMaillistsSyncView,
        AdminOrganizationServicesSubscribersView,
        AdminOrganizationDomainsView,
        AdminOrganizationChangeOwnerView,
        AdminOrganizationsDomainOwnershipView,
        AdminOrganizationFeaturesListView,
        AdminOrganizationFeaturesManageView,
        AdminPartnersList,
        AdminOrganizationWhitelistManageView,
    )
    from intranet.yandex_directory.src.yandex_directory.admin.views.users import (
        AdminUserDetailView,
        AdminAdminDetailView,
    )
    from intranet.yandex_directory.src.yandex_directory.admin.views.allowed_routes import (
        AdminAllowedRoutesView,
        ROLES_TO_METHODS,
    )
    from intranet.yandex_directory.src.yandex_directory.admin.views.tasks import (
        AdminTasksListView,
        AdminTaskDetailView,
    )
    from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
    from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
        scopes_required,
        internal as internal_decorator,
        no_permission_required,
    )
    from intranet.yandex_directory.src.yandex_directory.core.permission.internal_permissions import (
        all_internal_admin_permissions,
        all_internal_bizdev_permissions,
        all_internal_support_permissions,
        all_internal_assessor_permissions,
    )
    from intranet.yandex_directory.src.yandex_directory.core.idm import ADMIN_ROLES

    app.ADMIN_ROUTES = [
        ('/admin/organizations/', AdminOrganizationsListView),
        ('/admin/organizations/<int:org_id>/', AdminOrganizationDetailView),
        ('/admin/organizations/<int:org_id>/owner/', AdminOrganizationChangeOwnerView),
        ('/admin/organizations/<int:org_id>/services/', AdminOrganizationServicesListView),
        ('/admin/organizations/<int:org_id>/services/<service_slug>/', AdminOrganizationServicesDetailView),
        ('/admin/organizations/<int:org_id>/services/<service_slug>/enable/', AdminOrganizationServicesEnableView),
        ('/admin/organizations/<int:org_id>/services/<service_slug>/disable/', AdminOrganizationServicesDisableView),
        ('/admin/organizations/<int:org_id>/type/', AdminOrganizationChangeType),
        ('/admin/organizations/<int:org_id>/subscription_plan/', AdminOrganizationChangeSubscriptionPlanView),
        ('/admin/organizations/<int:org_id>/domains/<domain_name>/verify/', AdminOrganizationVerifyDomainView),
        ('/admin/organizations/total_count/', AdminOrganizationTotalCount),
        ('/admin/organizations/<int:org_id>/admins/', AdminOrganizationAdminsListView),
        ('/admin/organizations/<int:org_id>/domains/', AdminOrganizationDomainsView),
        ('/admin/organizations/<int:org_id>/domains/<domain_name>/', AdminOrganizationsDomainOwnershipView),
        ('/admin/organizations/<int:org_id>/tasks/', AdminTasksListView),
        ('/admin/organizations/<int:org_id>/services/<slug>/subscribers/', AdminOrganizationServicesSubscribersView),
        ('/admin/organizations/<int:org_id>/actions/', AdminOrganizationsActionsListView),
        ('/admin/organizations/<int:org_id>/staff/', AdminOrganizationStaffListView),
        ('/admin/organizations/<int:org_id>/maillists/sync/', AdminOrganizationMaillistsSyncView),
        ('/admin/organizations/<int:org_id>/features/', AdminOrganizationFeaturesListView),
        ('/admin/organizations/<int:org_id>/features/<feature_slug>/<action>/', AdminOrganizationFeaturesManageView),
        ('/admin/organizations/<int:org_id>/whitelist/', AdminOrganizationWhitelistManageView),
        ('/admin/organizations/<int:org_id>/access-restores/', AdminOrganizationAccessRestoreListView),
        ('/admin/organizations/<int:org_id>/block/', OrganizationBlockView),
        ('/admin/organizations/<int:org_id>/unblock/', OrganizationUnblockView),
        ('/admin/allowed-routes/', AdminAllowedRoutesView),
        ('/admin/tasks/names/', TaskNamesView),
        ('/admin/users/<int:user_id>/', AdminUserDetailView),
        ('/admin/users/<int:user_id>/contacts/', UserContactsView),
        ('/admin/tasks/<task_id>/', AdminTaskDetailView),
        ('/admin/tasks/dependencies/<dependent_id>/', AdminTasksDependeciesListView),
        ('/admin/partners/', AdminPartnersList),
        ('/admin/user/', AdminAdminDetailView),
        ('/admin/resources/history/', ResourceHistoryView),
        ('/admin/organizations/<int:org_id>/services/tracker/licenses/', AdminTrackerLicensesView),
    ]

    from intranet.yandex_directory.src.yandex_directory.core.views.idm import (
        AddRoleIDMView,
        RemoveRoleIDMView,
        InfoIDMView,
    )
    from intranet.yandex_directory.src.yandex_directory.core.views.idm.base import wrap_with_service_required

    app.IDM_ROUTES = [
        ('/idm/<service_slug>/add_role', AddRoleIDMView),
        ('/idm/<service_slug>/remove_role', RemoveRoleIDMView),
        ('/idm/<service_slug>/info', InfoIDMView),
    ]

    # добавим подмену стандартных ошибок на наши
    app.register_error_handler(404, json_error_not_found)

    def wrap_with_scopes_required(view_class, scopes):
        """
        Заворачиваем все http-методы класса view_class
        в декоратор scopes_required c заданным списком скоупов scopes
        :param view_class:
        :type view_class: View
        :param scopes:
        :type scopes: list of strings
        """
        for name, method in view_methods(view_class):
            method = scopes_required(scopes)(method)
            setattr(view_class, name, method)

    def wrap_with_internal(view_class):
        """
        Заворачиваем http-методы класса view_class
        в декоратор internal (так как админка - внутренний сервис)
        :param view_class:
        :type view_class: View
        """
        for name, method in view_methods(view_class):
            method = internal_decorator(method)
            setattr(view_class, name, method)

    def wrap_with_no_permission_required(view_class):
        """
        Заворачиваем все http-методы класса view_class
        в декоратор no_permission_required
        :param view_class:
        :type view_class: View
        """
        for name, method in view_methods(view_class):
            method = no_permission_required(method)
            setattr(view_class, name, method)

    def fill_roles_to_methods_map(view_class, uri):
        for name, method in view_methods(view_class):
            if hasattr(method, 'permissions'):
                _add_route(all_internal_admin_permissions, method, name, ADMIN_ROLES.admin, uri)
                _add_route(all_internal_support_permissions, method, name, ADMIN_ROLES.support, uri)
                _add_route(all_internal_assessor_permissions, method, name, ADMIN_ROLES.assessor, uri)
                _add_route(all_internal_bizdev_permissions, method, name, ADMIN_ROLES.bizdev, uri)

    def _add_route(permissions, method_func, method_name, role, uri):
        if set(method_func.permissions).issubset(set(permissions)):
            ROLES_TO_METHODS[role].append({'method': method_name, 'route': uri})

    def enable_admin_api_if_needed():
        if app.config['ENABLE_ADMIN_API']:
            initialize_admin_routes()

    app.enable_admin_api_if_needed = enable_admin_api_if_needed

    def initialize_admin_routes():
        """
        Добавляет в приложение роуты админки
        """
        if 'admin' not in app.blueprints:
            with versioned_blueprint(app, 'admin', __name__) as bp:
                for uri, view_class in app.ADMIN_ROUTES:
                    wrap_with_scopes_required(view_class, [scope.internal_admin])
                    wrap_with_internal(view_class)
                    fill_roles_to_methods_map(view_class, uri)
                    view_func = getattr(view_class, 'as_view')(view_class.__name__)
                    bp.add_url_rule(uri, view_func=view_func)

    enable_admin_api_if_needed()
    enable_swagger_for(app)

    from intranet.yandex_directory.src.yandex_directory.access_restore.views.restore import (
        RestoreListView,
        RestoreDetailView,
        MaskOwnerLoginView,
    )
    from intranet.yandex_directory.src.yandex_directory.access_restore.views.webmaster import (
        RestoreOwnershipView,
        RestoreVerifyView,
    )

    app.RESTORE_ROUTES = [
        ('/restore/<restore_id>/', RestoreDetailView),
        ('/restore/', RestoreListView),
        ('/restore/<restore_id>/ownership/', RestoreOwnershipView),
        ('/restore/<restore_id>/ownership/verify/', RestoreVerifyView),
        ('/restore/current-owner/<domain>/', MaskOwnerLoginView),
    ]

    def initialize_restore_routes():
        """
        Добавляет в приложение роуты для восстанволения доступа
        """
        if 'restore' not in app.blueprints:
            with versioned_blueprint(app, 'restore', __name__) as bp:
                for uri, view_class in app.RESTORE_ROUTES:
                    wrap_with_internal(view_class)
                    view_func = getattr(view_class, 'as_view')(view_class.__name__)
                    bp.add_url_rule(uri, view_func=view_func)

    initialize_restore_routes()

    def initialize_idm_routes():
        """
        Добавляет в приложение роуты для IDM
        """
        if 'idm' not in app.blueprints:
            with versioned_blueprint(app, 'idm', __name__) as bp:
                for uri, view_class in app.IDM_ROUTES:
                    wrap_with_scopes_required(view_class, [scope.connect_idm_application])
                    wrap_with_internal(view_class)
                    wrap_with_no_permission_required(view_class)
                    wrap_with_service_required(view_class)

                    view_func = getattr(view_class, 'as_view')(view_class.__name__)
                    bp.add_url_rule(uri, view_func=view_func)

    initialize_idm_routes()

    app.wsgi_app = server_port_fixer(
        VersionDispatcher(app=app.wsgi_app)
    )
    app.cached_view_methods = build_cached_view_methods_map()

    app.requests = create_requests_session(app.config['YANDEX_ALL_CA'])
    setup_zora(app)

    setup_metrika_client(app)
    setup_direct_client(app)
    setup_domenator_client(app)

    setup_yandexsprav_client(app)
    setup_alice_b2b_client(app)

    setup_partner_disk_client(app)
    # Надо понять откуда брался этот атрибут
    app.testing = True
    return app
