# -*- coding: utf-8 -*-
import urllib.parse
from functools import partial
from retrying import retry
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.common.utils import url_join
from intranet.yandex_directory.src.yandex_directory.common import http_client
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import error_log, default_log


class SsoConfigServiceError(APIError):
    code = 'sso_config_service_unknown_error'
    message = 'Error in SSO Config Service API'
    description = 'Неизвестная ошибка сервиса управления конфигурациями SSO.'


class SsoConfigServiceNotFoundError(SsoConfigServiceError):
    code = 'sso_config_service_not_found_error'
    message = 'Config not found in SSO Config Service'
    description = 'Настройки SSO не найдены.'


class SsoConfig(object):
    BINDING_HTTP_POST = 'urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST'
    BINDING_HTTP_REDIRECT = 'urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect'

    def __init__(self, **kwargs):
        self.id = kwargs.get('id')
        self.domain_ids = kwargs.get('domain_ids')
        self.entity_id = kwargs.get('entity_id')
        self.single_sign_on_service_url = kwargs.get('single_sign_on_service_url')
        self.single_sign_on_service_binding = kwargs.get('single_sign_on_service_binding')
        self.single_logout_service_url = kwargs.get('single_logout_service_url')
        self.single_logout_service_binding = kwargs.get('single_logout_service_binding')
        self.x509_cert_new = kwargs.get('x509_cert_new')
        self.x509_cert_old = kwargs.get('x509_cert_old')
        self.client_id = kwargs.get('client_id')
        self.enabled = kwargs.get('enabled')

    def copy(self):
        return SsoConfig(
            id=self.id,
            domain_ids=self.domain_ids,
            entity_id=self.entity_id,
            single_sign_on_service_url=self.single_sign_on_service_url,
            single_sign_on_service_binding=self.single_sign_on_service_binding,
            single_logout_service_url=self.single_logout_service_url,
            single_logout_service_binding=self.single_logout_service_binding,
            x509_cert_new=self.x509_cert_new,
            x509_cert_old=self.x509_cert_old,
            client_id=self.client_id,
            enabled=self.enabled,
        )

    @staticmethod
    def from_json(data):
        return SsoConfig(
            id=data['config_id'],
            domain_ids=data['domain_ids'],
            entity_id=data['entity_id'],
            single_sign_on_service_url=data['saml_config']['single_sign_on_service']['url'],
            single_sign_on_service_binding=data['saml_config']['single_sign_on_service']['binding'],
            single_logout_service_url=data['saml_config']['single_logout_service']['url'],
            single_logout_service_binding=data['saml_config']['single_logout_service']['binding'],
            x509_cert_new=data['saml_config']['x509_cert']['new'],
            x509_cert_old=data['saml_config']['x509_cert']['old'],
            client_id=data['oauth_config']['client_id'],
            enabled=data.get('enabled', True),
        )

    def to_json(self):
        return {
            'saml_config': {
                'single_sign_on_service': {
                    'url': self.single_sign_on_service_url,
                    'binding': self.single_logout_service_binding,
                },
                'single_logout_service': {
                    'url': self.single_logout_service_url,
                    'binding': self.single_logout_service_binding,
                },
                'x509_cert': {
                    'new': self.x509_cert_new,
                    'old': self.x509_cert_old,
                }
            },
            'oauth_config': {
                'client_id': self.client_id,
            },
            'enabled': self.enabled,
        }

    def __str__(self):
        return f'SsoConfig(id={self.id},entity_id={self.entity_id},sign_url={self.single_sign_on_service_url},client_id={self.client_id},domain_ids={self.domain_ids})'


@retry(stop_max_attempt_number=3, retry_on_exception=retry_http_errors('sso_config'))
def _make_request(method, uri, get_params=None, post_params=None):
    ticket = tvm.tickets['sso_config']
    headers = {
        'X-Ya-Service-Ticket': ticket,
    }

    if get_params is None:
        get_params = {}
    get_params['namespace'] = app.config['SSO_CONFIG_NAMESPACE']

    domain_ids = get_params.get('domain_id')
    if 'domain_id' in get_params:
        del get_params['domain_id']

    url = url_join(app.config['SSO_CONFIG_HOST'], uri, query_params=get_params)
    if domain_ids is not None:
        url += '&' + '&'.join(['domain_id=' + urllib.parse.quote_plus(str(x)) for x in domain_ids])

    response = http_client.request(method, url, json=post_params, headers=headers)
    if response.status_code >= 400:
        if response.status_code == 404:
            raise SsoConfigServiceNotFoundError()
        error_log.error('SsoConfigService response: %s' % response.content)
        raise SsoConfigServiceError()

    return response.json()


_get = partial(_make_request, 'get')
_post = partial(_make_request, 'post')
_put = partial(_make_request, 'put')
_delete = partial(_make_request, 'delete')


def get_config(domain_id=None, entity_id=None):
    try:
        if domain_id is not None:
            response = _get('/1/config/by_domain_id/%s/' % domain_id)
        elif entity_id is not None:
            response = _get('/1/config/by_entity_id/%s/' % urllib.parse.quote_plus(entity_id))
        else:
            raise RuntimeError('DomainId and EntityId is none')
    except SsoConfigServiceNotFoundError:
        return None

    return SsoConfig.from_json(response)


def create_config(config):
    default_log.info(f'Create config {config}')
    return _post(
        '/1/config/',
        get_params={'domain_id': config.domain_ids, 'entity_id': config.entity_id},
        post_params=config.to_json(),
    )


def update_config(config):
    default_log.info(f'Update config {config}')
    return _put(
        '/1/config/by_config_id/%s/' % config.id,
        get_params={'domain_id': config.domain_ids, 'entity_id': config.entity_id},
        post_params=config.to_json(),
    )


def delete_config(config):
    default_log.info(f'Delete config {config}')
    return _delete(
        '/1/config/by_config_id/%s/' % config.id
    )
