from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection
from intranet.yandex_directory.src.yandex_directory.core.models import UserModel, DomainModel
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.utils import add_existing_user_with_lock, is_domain_uid, \
    prepare_data_from_blackbox_response
from intranet.yandex_directory.src.yandex_directory.core.utils.users.dismiss import dismiss_user
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import default_log, error_log
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationSsoSettingsModel
from intranet.yandex_directory.src.yandex_directory.sso.utils import disable
from intranet.yandex_directory.src.yandex_directory.core.actions import action_user_modify
from intranet.yandex_directory.src import blackbox_client
from intranet.yandex_directory.src.yandex_directory.common.utils import to_lowercase


class DisableSsoTask(Task):
    singleton = True

    def do(self, org_id):
        disable(self.main_connection, org_id)


class SyncSsoOrgsTask(Task):
    singleton = True
    org_id_is_required = False

    def do(self):
        metadata = self.get_metadata() or {}
        last_org_id = metadata.get('last_org_id', 0)

        org_ids = OrganizationSsoSettingsModel(self.main_connection)\
            .filter(enabled=True, org_id__gt=last_org_id)\
            .order_by('org_id')\
            .limit(app.config['SYNC_SSO_ORGS_TASK_CHUNK_SIZE'])\
            .scalar('org_id')

        for org_id in org_ids:
            try:
                SyncSsoOrgTask(self.main_connection).delay(org_id=org_id)
            except DuplicatedTask:
                default_log.info(f'DuplicatedTask: SyncSsoOrgTask already in queue for {org_id}')

        if len(org_ids) == app.config['SYNC_SSO_ORGS_TASK_CHUNK_SIZE']:
            self.set_metadata({'last_org_id': org_ids[-1]})
            self.defer(countdown=1)


class SyncSsoOrgTask(Task):
    singleton = True
    lock_ttl = 1200

    count_executed_operations = 0

    def do(self, org_id):
        if not OrganizationSsoSettingsModel(self.main_connection).is_enabled(org_id):
            default_log.info('Organization not sso')
            return

        domain = DomainModel(self.main_connection).get_master(org_id=org_id)

        bb_org_uids = self._get_bb_uids(domain['name'])
        connect_org_uids = self._get_connect_uids(org_id)

        need_create_uids = bb_org_uids - connect_org_uids
        default_log.info('Found uids for create: %s' % need_create_uids)

        need_dismiss_uids = connect_org_uids - bb_org_uids
        default_log.info('Found uids for dismiss: %s' % need_dismiss_uids)

        need_sync_uids = connect_org_uids.intersection(bb_org_uids)
        default_log.info('Found uids for sync: %s' % need_sync_uids)

        has_error = False

        for uid in need_create_uids:
            has_error = self._create_user(org_id, uid) or has_error

        for uid in need_dismiss_uids:
            has_error = self._dismiss_user(org_id, uid) or has_error

        if need_sync_uids:
            fields = [
                blackbox_client.FIELD_LOGIN,
                blackbox_client.FIELD_FIRSTNAME,
                blackbox_client.FIELD_LASTNAME,
                ('userinfo.sex.uid', 'sex'),
                ('userinfo.birth_date.uid', 'birth_date'),
            ]
            bb_userinfo = {int(x['uid']): x for x in app.blackbox_instance.batch_userinfo(list(need_sync_uids), dbfields=fields, emails='getdefault')}
            connect_userinfo = {
                x['id']: x
                for x
                in UserModel(self.main_connection)
                   .filter(org_id=org_id, id=need_sync_uids)
                   .fields('org_id', 'nickname', 'name', 'email', 'gender', 'birthday')
                   .all()
            }

            for uid in need_sync_uids:
                has_error = self._sync_user(bb_userinfo[uid], connect_userinfo[uid]) or has_error

        if not has_error:
            OrganizationSsoSettingsModel(self.main_connection).update_last_sync_date(org_id)
        else:
            raise RuntimeError('Has errors on sync users')

    def _get_bb_uids(self, domain_name):
        from intranet.yandex_directory.src.blackbox_client import IS_MAILLIST_ATTRIBUTE

        uids = set()

        uids_info = app.blackbox_instance.batch_userinfo(
            app.blackbox_instance.account_uids(domain=domain_name),
            attributes=[IS_MAILLIST_ATTRIBUTE]
        )
        for userinfo in uids_info:
            uid = userinfo['uid']

            if userinfo['attributes'].get(IS_MAILLIST_ATTRIBUTE, '0') == '1':
                default_log.info('User %s is maillist' % uid)
                continue

            uids.add(int(uid))

        default_log.info('Found uids in blackbox: %s' % uids)
        return uids

    def _get_connect_uids(self, org_id):
        uids = [uid for uid in UserModel(self.main_connection).filter(org_id=org_id).scalar('id') if is_domain_uid(uid)]
        default_log.info('Found uids in connect: %s' % uids)

        return set(uids)

    def _increment_operations(self):
        if self.count_executed_operations > app.config['SYNC_SSO_ORG_TASK_MAX_OPERATIONS']:
            self.defer(countdown=1)
        self.count_executed_operations += 1

    def _create_user(self, org_id, uid):
        self._increment_operations()
        with get_meta_connection(for_write=True) as meta_connection:
            default_log.info('Create %s user' % uid)
            try:
                add_existing_user_with_lock(
                    meta_connection,
                    self.main_connection,
                    org_id,
                    uid,
                    is_sso=True,
                    set_org_id_sync=True,
                )
            except:
                error_log.exception(f'Error on create user {uid}', exc_info=True)
                return True

        return False

    def _dismiss_user(self, org_id, uid):
        self._increment_operations()
        with get_meta_connection(for_write=True) as meta_connection:
            default_log.info('Dismiss %s user' % uid)
            try:
                dismiss_user(
                    meta_connection,
                    self.main_connection,
                    org_id,
                    uid,
                    author_id=0,
                    skip_passport=True,
                )
            except:
                error_log.exception(f'Error on dismiss user {uid}', exc_info=True)
                return True

        return False

    def _sync_user(self, bb_userinfo, connect_userinfo):
        uid = int(bb_userinfo['uid'])
        try:
            (login, first, last, gender, birthday, email, cloud_uid) = prepare_data_from_blackbox_response(bb_userinfo)
            nickname = login.split('@')[0]
            has_diff = False

            def _check_field(field, connect_value, bb_value):
                nonlocal has_diff
                if bb_value != connect_value:
                    default_log.info(f'{field} is different for user {uid}. BB: {bb_value}. Connect: {connect_value}')
                    has_diff = True

            _check_field('firstname', connect_userinfo['name'].get('first'), (first or nickname))
            _check_field('lastname', connect_userinfo['name'].get('last'), (last or nickname))
            _check_field('gender', connect_userinfo['gender'], gender)
            _check_field('birthday', connect_userinfo['birthday'], birthday)

            if not has_diff:
                return False

            changed_data = {
                'name': {'first': first or nickname, 'last': last or nickname},
                'gender': gender,
                'birthday': birthday,
            }
            UserModel(self.main_connection).update_one(
                filter_data={'id': uid},
                update_data=changed_data,
            )
            user = UserModel(self.main_connection).get(uid, connect_userinfo['org_id'])
            action_user_modify(
                self.main_connection,
                org_id=connect_userinfo['org_id'],
                author_id=0,
                object_value=user,
                old_object=user,
            )
        except:
            error_log.exception(f'Error on sync user {uid}', exc_info=True)
            return True

        return False

