from flask import g

from intranet.yandex_directory.src.yandex_directory.core.features import is_feature_enabled, SSO_AVAILABLE
from intranet.yandex_directory.src.yandex_directory.common.utils import get_domain_info_from_blackbox
from intranet.yandex_directory.src.yandex_directory.core.models import DomainModel
from intranet.yandex_directory.src.yandex_directory.sso.config_service import client
from intranet.yandex_directory.src.yandex_directory.sso.exceptions import ConfigNotFoundException
from intranet.yandex_directory.src.yandex_directory.core.actions import action_sso_settings_modify
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationSsoSettingsModel
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection, get_main_connection, \
    get_shard_numbers
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationMetaModel
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import default_log
from intranet.yandex_directory.src.yandex_directory.domain.tasks import DomainGlobalLogoutTask


def is_sso_available(meta_connection, org_id):
    return is_feature_enabled(meta_connection, org_id, SSO_AVAILABLE)


def enable(main_connection, org_id):
    _change_enable(main_connection, org_id, True)


def disable(main_connection, org_id):
    _change_enable(main_connection, org_id, False)


def _change_enable(main_connection, org_id, status):
    config = get_config(main_connection, org_id)
    if config is None:
        raise ConfigNotFoundException()
    new_config = config.copy()
    new_config.enabled = status
    update_config(main_connection, config, new_config)


def get_config(main_connection, org_id):
    domain = DomainModel(main_connection).get_master(org_id)
    return get_config_by_domain_id(get_domain_info_from_blackbox(domain['name'])['domain_id'])


def get_config_by_domain_id(domain_id):
    return client.get_config(domain_id=domain_id)


def get_config_by_entity_id(entity_id):
    return client.get_config(entity_id=entity_id)


def create_config(main_connection, org_id, config):
    client.create_config(config)
    OrganizationSsoSettingsModel(main_connection).insert_or_update(org_id, config.enabled, bool(config.client_id))
    action_sso_settings_modify(
        main_connection,
        org_id=org_id,
        author_id=_get_author_uid(),
        object_value=config.to_json(),
    )


def update_config(main_connection, old_config, new_config):
    client.update_config(new_config)

    domains = []
    for domain_id in new_config.domain_ids:
        for hosted_domain in app.blackbox_instance.hosted_domains(domain_id=domain_id)['hosted_domains']:
            domains.append(hosted_domain['domain'])
    default_log.info(f'Found {domains} from config domain ids')

    domains = DomainModel(None).find_all(filter_data={'name': domains, 'owned': True}, fields=['org_id'])
    org_ids = [x['org_id'] for x in domains]
    default_log.info(f'Found {org_ids} for update config')

    with get_meta_connection() as meta_connection:
        for shard, org_ids in OrganizationMetaModel(meta_connection).get_orgs_by_shards(*org_ids).items():
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                for org_id in org_ids:
                    OrganizationSsoSettingsModel(main_connection).insert_or_update(org_id, new_config.enabled, bool(new_config.client_id))
                    if new_config.enabled:
                        OrganizationSsoSettingsModel(main_connection).update_last_sync_date(org_id)
                    action_sso_settings_modify(
                        main_connection,
                        org_id=org_id,
                        author_id=_get_author_uid(),
                        object_value=new_config.to_json(),
                        old_object=None if old_config is None else old_config.to_json(),
                    )

    if not new_config.enabled:
        with get_main_connection(shard=get_shard_numbers()[0], for_write=True) as main_connection:
            for domain in domains:
                DomainGlobalLogoutTask(main_connection).delay(domain=domain['name'])


def delete_domain_from_config(domain_id, config):
    config.domain_ids.remove(domain_id)

    if config.domain_ids:
        client.update_config(config)
    else:
        client.delete_config(config)


def _get_author_uid():
    if hasattr(g, 'user') and hasattr(g.user, 'passport_uid'):
        return g.user.passport_uid

    return None


class UserPatchDataValidator:
    @staticmethod
    def has_password(data):
        fields = ['password']
        return any(x in data for x in fields)

    @staticmethod
    def has_personal_info(data):
        fields = ['name', 'birthday', 'gender']
        return any(x in data for x in fields)

    @staticmethod
    def has_unblock(data):
        return 'is_enabled' in data and data['is_enabled'] is True

    @staticmethod
    def has_block(data):
        return 'is_enabled' in data and data['is_enabled'] is False
