from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.auth.decorators import internal, scopes_required, \
    permission_required, requires
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from .exceptions import ConfigNotFoundException
from .utils import is_sso_available, enable, disable, get_config, get_config_by_domain_id, create_config, update_config, \
    get_config_by_entity_id, delete_domain_from_config
from ..common import schemas
from ..common.exceptions import MasterDomainNotFound
from ..common.utils import json_response, json_error_not_found, json_error_invalid_value, get_domain_info_from_blackbox, \
    json_error
from ..core.models import DomainModel, UserModel
from ..core.permission.permissions import sso_permissions
from .config_service.client import SsoConfig
from ..directory_logging.logger import default_log
from ..swagger import uses_schema
from flask import g

SSO_SETTINGS_SCHEMA = {
    'title': 'Save sso settings',
    'type': 'object',
    'properties': {
        'entity_id': schemas.STRING,
        'single_sign_on_service_url': schemas.STRING,
        'certs': schemas.LIST_OF_STRINGS,
        'client_id': schemas.STRING_OR_NULL,
    },
    'required': ['entity_id', 'single_sign_on_service_url', 'certs'],
    'additionalProperties': True
}


class SsoAllowedView(View):
    @internal
    @requires(org_id=True, user=True)
    @permission_required([sso_permissions.read_sso_settings])
    @scopes_required([scope.read_sso_settings, scope.write_sso_settings])
    def get(self, meta_connection, main_connection):
        if not is_sso_available(meta_connection, g.org_id):
            return json_response({
                'allowed': False,
                'reason': 'not available',
            })

        try:
            DomainModel(main_connection).get_master(g.org_id)
        except MasterDomainNotFound:
            return json_response({
                'allowed': False,
                'reason': 'no domain',
            })

        aliases = DomainModel(main_connection).get_aliases(g.org_id, owned=None)
        if len(aliases) > 0:
            return json_response({
                'allowed': False,
                'reason': 'has alias domains',
            })

        users_count = UserModel(main_connection).filter(org_id=g.org_id, is_pdd=True, is_sso=False).count()
        if users_count > 0:
            return json_response({
                'allowed': False,
                'reason': 'has pdd accounts',
            })

        return json_response({
            'allowed': True,
        })


class SsoSettingsView(View):
    @internal
    @requires(org_id=True, user=True)
    @permission_required([sso_permissions.read_sso_settings])
    @scopes_required([scope.read_sso_settings, scope.write_sso_settings])
    def get(self, _, main_connection):
        config = get_config(main_connection, g.org_id)
        if config is None:
            return json_error_not_found()

        certs = []
        if config.x509_cert_new:
            certs.append(config.x509_cert_new)
        if config.x509_cert_old:
            certs.append(config.x509_cert_old)

        return json_response({
            'entity_id': config.entity_id,
            'single_sign_on_service_url': config.single_sign_on_service_url,
            'client_id': config.client_id,
            'certs': certs,
            'enabled': config.enabled,
        })

    @internal
    @requires(org_id=True, user=True)
    @permission_required([sso_permissions.write_sso_settings])
    @scopes_required([scope.write_sso_settings])
    @uses_schema(SSO_SETTINGS_SCHEMA)
    def put(self, _, main_connection, data):
        domain = DomainModel(main_connection).get_master(g.org_id)
        domain_id = int(get_domain_info_from_blackbox(domain['name'])['domain_id'])

        config_by_domain_id = get_config_by_domain_id(domain_id)
        default_log.info(f'Found config by domain id: {config_by_domain_id}')
        config_by_entity_id = get_config_by_entity_id(data.get('entity_id'))
        default_log.info(f'Found config by entity id: {config_by_entity_id}')

        x509_cert_new, x509_cert_old = self._get_certs(data)
        if config_by_entity_id is not None and domain_id not in config_by_entity_id.domain_ids:
            has_error = False

            x509_cert_old_1 = config_by_entity_id.x509_cert_old.strip().replace('\n', '')
            x509_cert_old_2 = x509_cert_old.strip().replace('\n', '')
            if x509_cert_old_1 != x509_cert_old_2:
                default_log.info(
                    f'Old cert is different:\nME:\n{x509_cert_old_1}\nYOU:\n{x509_cert_old_2}'
                )
                has_error = True

            x509_cert_new_1 = config_by_entity_id.x509_cert_new.strip().replace('\n', '')
            x509_cert_new_2 = x509_cert_new.strip().replace('\n', '')
            if x509_cert_new_1 != x509_cert_new_2:
                default_log.info(
                    f'New cert is different:\nME:\n{x509_cert_new_1}\nYOU:\n{x509_cert_new_2}'
                )
                has_error = True

            if config_by_entity_id.single_sign_on_service_url.strip() != data.get('single_sign_on_service_url', '').strip():
                default_log.info(
                    f"Sign url is different:\nME:\n{config_by_entity_id.single_sign_on_service_url.strip()}\nYOU:\n{data.get('single_sign_on_service_url').strip()}"
                )
                has_error = True

            if config_by_entity_id.client_id.strip() != data.get('client_id', '').strip():
                default_log.info(
                    f"Client id is different:\nME:\n{config_by_entity_id.client_id.strip()}\nYOU:\n{data.get('client_id', '').strip()}"
                )
                has_error = True

            if has_error:
                return json_error(
                    409,
                    'config_exists',
                    'Config exists'
                )

        if config_by_domain_id is not None and config_by_domain_id.entity_id != data.get('entity_id'):
            delete_domain_from_config(domain_id, config_by_domain_id)

        if config_by_entity_id is None:
            config = SsoConfig(
                domain_ids=[domain_id],
                entity_id=data.get('entity_id'),
                single_sign_on_service_url=data.get('single_sign_on_service_url'),
                single_sign_on_service_binding=SsoConfig.BINDING_HTTP_REDIRECT,
                single_logout_service_url='',
                single_logout_service_binding=SsoConfig.BINDING_HTTP_REDIRECT,
                x509_cert_new=x509_cert_new,
                x509_cert_old=x509_cert_old,
                client_id=data.get('client_id'),
                enabled=True if config_by_domain_id is None else config_by_domain_id.enabled,
            )
            create_config(main_connection, g.org_id, config)

            return json_response(config.to_json(), status_code=201)
        else:
            new_config = config_by_entity_id.copy()
            new_config.entity_id = data.get('entity_id')
            new_config.single_sign_on_service_url = data.get('single_sign_on_service_url')
            new_config.x509_cert_new = x509_cert_new
            new_config.x509_cert_old = x509_cert_old
            new_config.client_id = data.get('client_id')
            if domain_id not in new_config.domain_ids:
                new_config.domain_ids.append(domain_id)
            update_config(main_connection, config_by_domain_id, new_config)

            return json_response(new_config.to_json(), status_code=201)

    def _get_certs(self, data):
        x509_cert_new = ""
        x509_cert_old = ""

        certs = data.get('certs')
        if len(certs) >= 1:
            x509_cert_new = certs[0]
        if len(certs) >= 2:
            x509_cert_old = certs[1]

        return x509_cert_new, x509_cert_old


class SsoActionsView(View):
    @internal
    @requires(org_id=True, user=True)
    @permission_required([sso_permissions.read_sso_settings])
    @scopes_required([scope.write_sso_settings])
    def put(self, _, main_connection, action):
        try:
            if action == 'enable':
                enable(main_connection, g.org_id)
            elif action == 'disable':
                disable(main_connection, g.org_id)
            else:
                return json_error_invalid_value('action')
        except ConfigNotFoundException:
            return json_error_not_found()

        return json_response({})
