# coding: utf-8
import os
import json
import commonmark
from collections import defaultdict
from frozendict import frozendict

from functools import wraps
from flask import send_from_directory, request, Response, redirect, g
from intranet.yandex_directory.src.yandex_directory.common.utils import validate_data_by_schema, json_error_bad_request
from intranet.yandex_directory.src.yandex_directory.auth.decorators import no_auth


def _process_doc(text):
    parser = commonmark.Parser()
    renderer = commonmark.HtmlRenderer()
    if isinstance(text, bytes):
        text = text.decode('utf-8')
    ast = parser.parse(text)
    return renderer.render(ast)


def enable_swagger_for(app):
    """
    В development/testing/production эти ручки отрабатывать не будут, т.к. будут спрятаны за nginx
    """
    @no_auth
    @app.route('/docs/')
    def redirect_to_index_html():
        return redirect('/docs/index.html', code=302)

    @no_auth
    @app.route('/docs/<path:path>')
    def serve_static_docs(path):
        base_path = app.config['BASE_PATH'] # this is a src dir
        docs_dir = os.path.join(
            os.path.dirname(base_path),
            'rst-docs',
            'build',
        )
        return send_from_directory(docs_dir, path)


def _schema_to_swagger(schema, api_version):
    """Преобразуем некоторые атрибуты JSON схемы, чтобы их понимал swagger.
    """
    from intranet.yandex_directory.src.yandex_directory.common import schemas

    # В новых версиях API мы отказались от интернациализации строк,
    # но в схемах их пока до конца не выпилили, и принимаем в новых ручках
    # как обычные строки, так и интернационализированные.
    # Однако мы не хотим, чтобы пользователи новых ручек использовали
    # интернационализированные строки, поэтому в документации новых
    # ручек их не показываем, пока заменяя в JSON схемах типы параметров.
    # Это можно будет убрать, когда мы окончательно избавимся от ручек < 6
    # версии и поправим сами JSON схемы.
    if api_version >= 6:
        schema = schemas.replace_i18n_strings_in_schema(schema)

    def remove_key(obj, key):
        if not isinstance(obj, dict):
            return obj
        new_obj = obj.copy()
        if key in new_obj:
            del new_obj[key]
        for k, value in list(new_obj.items()):
            new_obj[k] = remove_key(value, key)
        return new_obj

    schema_type = schema.get('type')
    if isinstance(schema_type, (list, tuple)):
        # иногда в схеме указывается несколько типов, например
        # ['object', 'null'] и тогда надо выбрать 'object'
        schema_type = [t for t in schema_type
                       if t != 'null'][0]

    if schema_type == 'array':
        swagger_schema = {
            'id': schema['title'],
            'type': schema_type,
            'items': schema['items'],
        }
    else:
        swagger_schema = {
            'id': schema['title'],
            'type': schema_type,
            'properties': schema.get('properties', {})
        }

        if 'required' in schema:
            swagger_schema['required'] = schema['required']
    swagger_schema = remove_key(swagger_schema, 'anyOf')
    return swagger_schema


def uses_schema(schema):
    """Декоратор для методов, которые принимают JSON в теле POST
    и этот JSON должен соответствовать заданной схеме.

    Дополнительно, в качестве первого параметра в декорируемую функцию
    передается уже распаршенное тело поста, a к метаданным функции добавляется
    JSON схема, и для она становится доступна для Swagger.

    Если указать body_required = False, то ручка сможет корректно работать
    даже если в неё не отправляют body. В этом случае схема не должна
    содержать required параметров, а в качесте data на вход метода поступит
    пустой словарь.
    """
    def decorator(func):
        @wraps(func)
        def wrapper(self, *args, **kwargs):

            # в ряде случаев возможно, что в request хранится какой-то мусор
            # нам подходит только вариант, когда есть содержимое, в котором
            # содержащийся тип является application/json
            if request.is_json and request.data:
                json_data = request.get_json()
                data = self.normalization(json_data, schema=schema)
            else:
                data = {}

            errors = None

            if not g.use_cloud_proxy:
                errors = validate_data_by_schema(data, schema=schema)
                # Мы замораживаем словарь с входными данными, чтобы
                # исключить его модификацию во вьюшках. Так легче гарантировать,
                # что в коде не будет сайд-эффектов.
                if isinstance(data, dict):
                    data = frozendict(data)

            if errors:
                response = {
                    'message': 'Schema validation error',
                    'code': 'schema_validation_error',
                    'params': {
                        'schema': schema,
                        'errors': errors,
                    }
                }

                return Response(
                    json.dumps(response),
                    status=422,
                    mimetype='application/json; charset=utf-8',
                )
            # data добавляется к остальным позиционным параметрам,,
            # потому что сначала обычно идут коннекты к базе
            args += (data,)

            return func(self, *args, **kwargs)
        wrapper.__schema__ = schema
        return wrapper
    return decorator


def uses_schema_for_get(schema):
    """Декоратор для методов, которые принимают аргументы в GET.
    Аргументы должны соответствовать схеме. Аргументы преобразуются в словарь и
    если возможно, конвертируются в integer
    """
    def decorator(func):
        @wraps(func)
        def wrapper(self, *args, **kwargs):

            data = defaultdict(list)
            for param_name, value in list(request.values.items()):
                value = value.split(',')
                for item in value:
                    item = item.strip()
                    try:
                        item = int(item)
                        data[param_name].append(item)
                    except ValueError:
                        if item:
                            data[param_name].append(item)

                if len(data[param_name]) == 1:
                    data[param_name] = data[param_name][0]

            errors = validate_data_by_schema(data, schema=schema)
            if errors:
                response = {
                    'message': 'Validation error',
                    'schema': schema,
                    'errors': errors
                }

                return Response(
                    json.dumps(response),
                    status=422,
                    mimetype='application/json; charset=utf-8',
                )
            return func(self, *args, **kwargs)

        wrapper.__schema__ = schema
        return wrapper
    return decorator


def uses_out_schema(schema):
    """
    Декоратор, задающий схему выходных данных для метода
    Args:
        schema (dict): схема данных, которые должны быть в response.data
    """
    def decorator(view_func):
        setattr(view_func, 'out_schema', schema)

        @wraps(view_func)
        def _wrapped_view(view_instance, *args, **kwargs):
            return view_func(view_instance, *args, **kwargs)
        return _wrapped_view
    return decorator
