# coding: utf-8

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    create_domain_in_passport,
    ensure_list_or_tuple,
    log_service_response,
    utcnow,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import disable_domain_in_organization
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DomainModel,
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.sms.tasks import sms_domain_confirmed
from intranet.yandex_directory.src.yandex_directory.core.tasks.tasks import UnblockDomainsTask
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    after_first_domain_became_confirmed,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import action_domain_occupy
from intranet.yandex_directory.src.yandex_directory.common.db import get_main_connection

# Полный список возможных типов можно посмотреть в документации:
# https://webmaster-internal.common.yandex.net/user/host/verification/verify.info#open(params-verificationType)
# но не все из них мы поддерживаем.
from intranet.yandex_directory.src.yandex_directory.common import http_client

ALLOWED_VERIFICATION_TYPES = (
    'DNS',
    'HTML_FILE',
    'META_TAG',
    'WHOIS',
    'DNS_DELEGATION',
    # Этот способ не надо показывать обычным пользователям, он используется лишь
    # для ПДД регистраторов.
    #
    # Внутри он включает в себя две проверки:
    #
    # – CNAME с yamail-c269887f812cb586.mydomain.ru должен вести на mail.yandex.ru
    # - Файла на http://mydomain.ru/c269887f812cb586.html
    #
    # достаточно, чтобы сработала любая из них.
    'PDD_EMU',
)
PUBLIC_VERIFICATION_TYPES = set(ALLOWED_VERIFICATION_TYPES) - {'DNS_DELEGATION', 'PDD_EMU'}


class WebmasterError(APIError):
    code = 'webmaster_internal_error'
    description = 'Внутренняя ошибка Вебмастера'
    log_level = 'WARNING'
    status_code = 424

    def __init__(self, webmaster_code, webmaster_message):
        super(WebmasterError, self).__init__()
        self.code = 'webmaster_' + webmaster_code.lower()
        self.message = webmaster_message


def get_errors_except_ignored(response, ignore_errors):
    errors = response.get('errors', [])

    if ignore_errors is not None:
        ignore_errors = ensure_list_or_tuple(ignore_errors)
        # Уберём ошибки, которые нужно игнорировать
        errors = [
            error
            for error in errors
            if error['code'] not in ignore_errors
        ]
    return errors


def make_inner_call(func, *args, **kwargs):
    """Вызывает функцию с заданными параметрами и проверяет ответ.

    Ответ должен быть словарём, который вернул нам Вебмастер.

    make_inner_call кидает исключение WebmasterError, если в ответе есть ошибки.

    В аргументе ignore_errors можно передать строковый код ошибки
    которую нужно игнорировать, или несколько ошибок списком.
    Он так же будет использоваться для определения того, какие ответы с каким
    уровнем логгировать. Если ошибка от вебастера указана в аргументе
    ignore_errors, то такой ответ будет логгироваться, как INFO.
    """
    webmaster_response = func(*args, **kwargs)
    ignore_errors = kwargs.get('ignore_errors', None)

    if ignore_errors is not True:
        errors = get_errors_except_ignored(webmaster_response, ignore_errors)

        if errors:
            # В исключении возвращаем код самой первой ошибки.
            # Вообще, я ни разу не видел, чтобы их было больше чем одна.
            first_error = errors[0]
            raise WebmasterError(
                first_error['code'],
                first_error.get('message') or 'No message',
            )
    # Чтобы можно было "чейнить" вызов этой функции
    return webmaster_response


def add(domain, admin_uid):
    # добавляем домен в вебмастер
    make_inner_call(
        _inner_add,
        domain,
        admin_uid,
        ignore_errors='ADD_HOST__HOST_ALREADY_ADDED',
    )


def add_registrar_domain(domain, admin_uid):
    """Добавляет домен от имени регистратора и запускает нужную проверку.
    """
    add(domain, admin_uid)
    verify(domain, admin_uid, 'PDD_EMU')


def _make_request(uri, **data):
    ignore_errors = data.pop('ignore_errors', None)

    if app.config['ENVIRONMENT'] == 'autotests':
        import traceback
        method_name = traceback.extract_stack()[-2][2]
        raise NotImplementedError('Please, mock webmaster.{0} method in the unittest'.format(method_name))

    ticket = tvm.get_tvm2_ticket(app.config['WEBMASTER_CLIENT_ID'])
    headers = {
        'X-Ya-Service-Ticket': ticket,
    }
    webmaster_base_url = app.config['WEBMASTER_URL']
    if not webmaster_base_url:
        raise RuntimeError('Webmaster base URL is unknown. Set WEBMASTER_URL variable')

    url = webmaster_base_url + uri
    response = http_client.request('post', url, json=data, headers=headers)

    def get_logger(data):
        errors = get_errors_except_ignored(data, ignore_errors)

        if errors:
            return log.error
        else:
            return log.info

    log_service_response('webmaster', 'POST', url, data, response, get_logger=get_logger)
    response.raise_for_status()

    return response.json()


def lock_dns_delegation(domain, admin_id, duration):
    """
    Закрепляем домен за пользователем на определённый срок.
    В течение этого срока другие пользователи не смогут выбрать подтверждение хоста
    через делегирование DNS хостингу Яндекса.
    """
    duration_minutes = duration / 60
    make_inner_call(
        _inner_lock_dns_delegation,
        domain,
        admin_id,
        duration_minutes,
    )


def verify(domain, admin_id, verification_type):
    # У вебмастера внутри все типы подтверждения написаны капслоком
    # поэтому прежде чем туда передавать что-то, надо способ подтверждения
    # привести к капслоку.
    original_verification_type = verification_type
    verification_type = verification_type.upper()

    # Проверим, что способ подтверждения указан правильно.
    assert verification_type in ALLOWED_VERIFICATION_TYPES, \
        'Verification type {0} is not supported'.format(original_verification_type)

    # Если подтверждаем через делегирование, то сначала закрепляем домен за пользователем на определённый срок
    if verification_type == 'DNS_DELEGATION':
        duration = app.config['LOCK_DNS_DELEGATION_DURATION']
        lock_dns_delegation(domain, admin_id, duration)

    return make_inner_call(
        _inner_verify,
        domain,
        admin_id,
        verification_type,
        ignore_errors='VERIFY_HOST__ALREADY_VERIFIED',
    )


def info(domain, admin_id, ignore_errors=None):
    """Здесь мы, через ручку info получаем данные о состоянии подтверждения домена.
    """
    response = make_inner_call(
        _inner_info,
        domain,
        admin_id,
        ignore_errors=ignore_errors,
    )
    return response


def is_verified(domain, admin_id):
    """
    Здесь мы, через ручку info проверяем, подтверждён ли домен в Вебмастере.
    Ручка должна вернуть данные со статусом VERIFIED или None, если домена нет в Вебмастере.
    """

    # Тестовые домены в специальной зоне мы всегда считаем подтверждёными.
    # Это позволит ассесорам проходить сценарии в которых нужно добавлять в организацию домен.
    autoapprove_suffix = app.config['AUTO_APPROVE_ZONE']
    if autoapprove_suffix and domain.endswith(autoapprove_suffix):
        return True

    response = info(domain, admin_id, ignore_errors='USER__HOST_NOT_ADDED')
    data = response.get('data') or {}
    status = data.get('verificationStatus')
    return status == 'VERIFIED'


def list_applicable(domain, admin_uid, only_public=True):
    """Возвращает список строк с именами методов подтверждения домена.

    Типа:

    [
        "META_TAG",
        "HTML_FILE",
        "DNS",
        "WHOIS",
        "DNS_DELEGATION"
    ]
    """
    response = make_inner_call(
        _inner_list_applicable,
        domain,
        admin_uid,
    )
    # Если домен не подтверждён, и проверка не запущена, то вебмастер отдаёт None
    # вместо словаря с данными
    data = response['data'] or {}
    methods = data.get('applicableVerifications', [])
    if only_public:
        methods = set(methods) & PUBLIC_VERIFICATION_TYPES
    return list(methods)


def set_domain_as_owned(
        meta_connection,
        main_connection,
        org_id,
        admin_id,
        domain_name,
):
    """
    Помечаем домен как подтвержденный
    в Директории и добаляем как алиас к мастер домену.
    """

    with log.fields(domain=domain_name):
        # Для блокировки/разблокировки домена мы берем отдельные коннекты, чтобы
        # возможные параллельные процессы видели, что домен временно заблокирован.
        shard = main_connection.engine.db_info['shard']
        with get_main_connection(shard=shard, for_write=True) as block_main:
            # перед началом подтверждения нужно заблокировать все домены в организации,
            # чтобы их нельзя было удалить или сменить мастер домен
            DomainModel(block_main).block(org_id)
        try:
            is_master = create_domain_in_passport(
                main_connection,
                org_id,
                domain_name,
                admin_id,
            )
            log.info('Setting domain as owned')
            data = {
                'validated_at': utcnow(),
                'owned': True,
            }

            # Обновим его статус в своей базе.
            domain_model = DomainModel(main_connection)
            domain_model.update_one(
                domain_name,
                org_id,
                data=data,
                force=True,
            )

            log.info('Add domain_occupy action')
            domain = DomainModel(main_connection).get(domain_name, org_id)
            action_domain_occupy(
                main_connection,
                org_id=org_id,
                author_id=admin_id,
                object_value=domain,
                old_object=None,
            )

            if is_master:
                after_first_domain_became_confirmed(
                    meta_connection,
                    main_connection,
                    org_id,
                    domain_name,
                    force=True,
                )
            DomainModel(main_connection).unblock(org_id)

        except Exception:
            # Если произошла ошибка, то значит разблокировать домен в основной транзакции
            # не получилось и надо сделать отдельную тасочку для разблокировки.
            # Делать это будем в отдельной транзакции,
            # так как текущая транзакция может быть нарушена.
            with get_main_connection(shard=shard, for_write=True) as unblock_main:
                UnblockDomainsTask(unblock_main).delay(org_id=org_id)
            raise


def set_domain_as_owned_in_connection(
        meta_connection,
        main_connection,
        org_id,
        admin_id,
        domain_name,
):
    """Делает то же самое что и set_domain_as_owned, но в рамках одной транзакции"""
    with log.fields(domain=domain_name):
        is_master = create_domain_in_passport(
            main_connection,
            org_id,
            domain_name,
            admin_id,
        )

        log.info('Setting domain as owned')
        data = {
            'validated_at': utcnow(),
            'owned': True,
        }

        # Обновим его статус в своей базе.
        domain_model = DomainModel(main_connection)
        domain_model.update_one(
            domain_name,
            org_id,
            data=data,
            force=True,
        )

        log.info('Add domain_occupy action')
        domain = DomainModel(main_connection).get(domain_name, org_id)
        action_domain_occupy(
            main_connection,
            org_id=org_id,
            author_id=admin_id,
            object_value=domain,
            old_object=None,
        )
        if is_master:
            after_first_domain_became_confirmed(
                meta_connection,
                main_connection,
                org_id,
                domain_name,
                force=True,
            )


def update_domain_state_if_verified(meta_connection,
                                    main_connection,
                                    org_id,
                                    admin_id,
                                    domain,
                                    send_sms=False):
    """Если домен стал подтверждённым в вебмастере, то обновим его состояние
       в нашей базе, а так же пошлём SMS админу о том, что домен
       подтвердился.

       Для доменов, которые добавлялись регистраторами, мы дополнительно
       перезапускаем проверку способои PDD_EMU, потому что пользователь
       не может это сделать сам, так как не участвует в процессе подтверждения:

       https://st.yandex-team.ru/WMC-6592#5d3736e029af1d001c1a20f4
    """

    domain_name = domain['name']
    is_already_owned = domain['owned']

    if not is_already_owned:
        # Проверим, может домен уже подтвердили
        owned = is_verified(domain_name, admin_id)

        # Если домен стал подтвержденным то:
        if owned:
            disable_domain_in_organization(domain_name, org_id, admin_id)

            set_domain_as_owned(
                meta_connection,
                main_connection,
                org_id,
                admin_id,
                domain_name,
            )
            if send_sms:
                # Пошлем про это sms админу организации.
                sms_domain_confirmed(
                    meta_connection,
                    main_connection,
                    org_id,
                    admin_id,
                    domain_name,
                )
        # Если домен не подтверждён
        else:
            # То узнаем каким способом он подтверждался и статус его проверки
            response = info(domain_name, admin_id, ignore_errors='USER__HOST_NOT_ADDED')
            data = response['data'] or {}
            verification_type = data.get('verificationType')
            verification_status = data.get('verificationStatus')
            # Если это домен от регистратора, то надо запустить проверку заново
            if verification_type == 'PDD_EMU' and verification_status == 'VERIFICATION_FAILED':
                verify(domain_name, admin_id, verification_type)


def reset(domain, for_user_id, user_id):
    """
    Делает HTTP запрос в Вебмастер для перепроверки прав на домен для указанного пользователя.
    В случае, если права были делегированы другим пользователем - отменить делегирование
    """
    return _inner_reset(
        domain=domain,
        for_user_id=for_user_id,
        user_id=user_id,
    )

#######################################################
# Внутренние хелперы, нужные для того, чтобы в тестах #
# можно было замокать вызов к API.                    #
#######################################################


def _inner_add(domain, admin_id, ignore_errors=None):

    """Делает HTTP запрос в Вебмастер, и возвращает все данные ответа
    в виде python структуры данных.

    Выделено в отдельную функцию для того, чтобы было удобнее мокать ответы
    Вебмастера в наших юнит-тестах.
    """
    return _make_request(
        '/user/domain/verification/add.json',
        domain=domain,
        userId=admin_id,
        ignore_errors=ignore_errors,
    )


def _inner_verify(domain, admin_id, verification_type, ignore_errors=None):
    return _make_request(
        '/user/domain/verification/verify.json',
        domain=domain,
        userId=admin_id,
        verificationType=verification_type,
        ignore_errors=ignore_errors,
    )


def _inner_info(domain, admin_id, ignore_errors=None):
    return _make_request(
        '/user/domain/verification/info.json',
        domain=domain,
        userId=admin_id,
        ignore_errors=ignore_errors,
    )


def _inner_list_applicable(domain, admin_id, ignore_errors=None):
    return _make_request(
        '/user/domain/verification/listApplicable.json',
        domain=domain,
        userId=admin_id,
        ignore_errors=ignore_errors,
    )


def _inner_lock_dns_delegation(domain, admin_id, duration_minutes, ignore_errors=None):
    return _make_request(
        '/user/host/verification/lockDnsDelegation.json',
        domain=domain,
        userId=admin_id,
        durationMinutes=duration_minutes,
        ignore_errors=ignore_errors,
    )


def _inner_reset(domain, for_user_id, user_id, ignore_errors=None):
    return _make_request(
        '/user/domain/verification/reset.json',
        domain=domain,
        forUserId=for_user_id,
        userId=user_id,
        ignore_errors=ignore_errors,
    )
