# -*- coding: utf-8 -*-
import enum

from intranet.yandex_directory.src import settings

from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.utils import raise_for_status
from .exceptions import (
    ZoraInvalidCodeError,
    ZoraInternalCodeError,
)


# see codes here: https://a.yandex-team.ru/arc/trunk/arcadia/robot/zora/gozora/internal/zerrors/errors.go
from ..common import http_client


class ZoraEnum(enum.IntEnum):
    INSECURE_CONNECTION = 1000
    DENIED_BY_FW = 1001
    TLS_HANDSHAKE = 1002
    TVM_AUTH_ERROR = 1003
    HOST_SPLIT = 1004
    DNS_RESOLVE = 1005
    DIAL_FAILED = 1006
    REMOTE_UPGRADE = 1007
    READ_RESPONSE = 1008
    RPS_LIMIT = 1009


INTERNAL_DOWNLOAD_ERRORS = [int(value) for value in ZoraEnum.__members__.values()]


class ZoraClient(object):
    def __init__(self):
        self.zora_template = settings.ZORA_HOST
        self.zora_timeout = 5

    def get(self, url, *args, **kwargs):
        self.update_kwargs(kwargs)
        response = http_client.request('get', url, *args, **kwargs)
        with log.fields(status_code=response.status_code, response_text=response.text, url=url):
            log.debug('Zora response')
        if response.status_code != 200:
            with log.fields(
                response_status_code=response.status_code,
                response_text=response.text
            ):
                zora_http_code = response.headers.get('X-Yandex-GoZora-Error-Code')
                if zora_http_code is not None:
                    try:
                        zora_http_code = int(zora_http_code)
                    except ValueError:
                        raise ZoraInvalidCodeError(zora_http_code)

                    if zora_http_code in INTERNAL_DOWNLOAD_ERRORS:
                        raise ZoraInternalCodeError(zora_http_code)

                raise_for_status(response)

        return response

    def update_kwargs(self, kwargs):
        zora_url = self.zora_template.format(tvm=tvm.tickets['gozora'])
        kwargs.setdefault('headers', {})
        kwargs['proxies'] = {
            'https': zora_url,
            'http': zora_url,
        }
        kwargs['verify'] = False
