# -*- coding: utf-8 -*-

import click
import json
import sys
import pytz
import hashlib
from dateutil.parser import parse
from datetime import datetime
import formic
import os
import zipfile
import gettext
import logging
from collections import defaultdict
from operator import itemgetter


import requests
from jinja2 import Environment, FileSystemLoader
from invoke import run, task
from common_tasks import (
    update_requirements as common_ur,
    make_dashed_aliases,
    check_if_dirty,
)


DOCKER_REGISTRY = 'registry.yandex.net'
DOCKER_REGISTRY_PROJECT = 'workspace/directory'
DOCKER_REGISTRY_PROJECT_URL = '%s/%s' % (DOCKER_REGISTRY, DOCKER_REGISTRY_PROJECT)

TANKER_OAUTH_TOKEN = 'AQAD-qJSJsLqAAAGzIEtxGw7y0KImBAHNQVICfM'
TANKER_API_BASE_URL = 'https://tanker-api.yandex-team.ru/'
TANKER_PROJECT_ID = 'directory'
TANKER_EMAIL_KEYSET = 'Email'
TANKER_SMS_KEYSET = 'Sms'
TANKER_API_MESSAGES_KEYSET = 'api_messages'

TANKER_KEYSET_IDS = [
    TANKER_EMAIL_KEYSET,
    TANKER_SMS_KEYSET,
    TANKER_API_MESSAGES_KEYSET,
]

DIRECTORY_LANGUAGES = {'ru', 'en'}

TEMPLATES_DIR = 'templates'
EMAIL_TEMPLATES = ['after_migration', 'welcome']

# Это нужно, чтобы manage.py не падал, когда запускается через
# invoke на OSX
os.environ['IGNORE_TICKET_PARSER'] = '1'


requests.packages.urllib3.disable_warnings()

@task
def compile_translations(ctx):
    """Convert *.po to *.mo files"""
    translations_root = os.path.join(
        os.path.dirname(os.path.abspath(__file__)),
        'i18n'
    )
    result = run('which msgfmt',  warn=True)
    if not result.ok:
        print('msgfmt is missing, try brew link gettext --force')
        sys.exit(1)
    # домен перевода соответствует имени keyset в нижнем регистре
    for keyset in TANKER_KEYSET_IDS:
        run('find  {path} -name {domain}.po -execdir msgfmt {domain}.po -o {domain}.mo \;'.format(
            path=translations_root,
            domain=keyset.lower(),
        ))


@task
def download_translations(ctx):
    """Download translations files (*.po) from Taker (https://tanker.yandex-team.ru)"""
    url_tmplate = TANKER_API_BASE_URL + 'keysets/po/?project-id={project}&keyset-id={keyset}&language={lang}'
    translations_root = os.path.join(
        os.path.dirname(os.path.abspath(__file__)),
        'i18n'
    )
    auth_headers = {'Authorization': 'OAuth {}'.format(TANKER_OAUTH_TOKEN)}
    for keyset in TANKER_KEYSET_IDS:
        for lang in DIRECTORY_LANGUAGES:
            url = url_tmplate.format(project=TANKER_PROJECT_ID, keyset=keyset, lang=lang)
            base_dir = os.path.join(translations_root, lang, 'LC_MESSAGES')
            if not os.path.exists(base_dir):
                os.makedirs(base_dir)
            # имя *.po и *.mo файл с переводами соответствует имени keyset в нижнем регистре
            po_file = os.path.join(base_dir, '{}.po'.format(keyset.lower()))
            with open(po_file, 'wb') as f:

                r = requests.get(url, headers=auth_headers, stream=True)

                if r.status_code != 200:
                    raise RuntimeError('Can\'t download "{lang}" translation file. (http status = {status_code})'.format(lang=lang, status_code=r.status_code))

                for chunk in r:
                    f.write(chunk)
            print('The "{lang}" translation file for keyset "{keyset}" successfully downloaded.'.format(lang=lang, keyset=keyset))
    compile_translations(ctx)


@task
def serve_docs(ctx, port=1080, rebuild=True):
    """Поднять вебсервер, который будет раздавать документацию на указанном порту."""
    port = int(port)

    def do_rebuild():
        build_docs(ctx, build_swagger_spec=False)

    if rebuild:
        do_rebuild()

    from livereload import Server

    server = Server()
    files = [
        'rst-docs/**/*.rst',
        'rst-docs/**/conf.py',
    ]
    for filepath in formic.FileSet(include=files, exclude=[]):
        server.watch(filepath, do_rebuild)

    server.serve(root='rst-docs/build', port=port, host='localhost')


@task
def build_events_docs(ctx):
    run('src/manage.py build-events-doc -o rst-docs/source/events.rst')


@task
def build_docs(ctx, build_swagger_spec=True):

    if build_swagger_spec:
        run('src/manage.py create-swagger-spec')

    build_command = 'sphinx-build -b html rst-docs/source rst-docs/build'
    run(build_command, pty=True)

    run('rm -fr rst-docs/build/static')
    run('mkdir rst-docs/build/static')
    run('cp -r docs/swagger rst-docs/build/static/')
    run('cp docs/jquery*.js rst-docs/build/static/')
    run('cp docs/index-with-spec*.js rst-docs/build/static/')
    run('cp docs/versions.js rst-docs/build/static/')
    run('cp docs/index.css rst-docs/build/static/')


@task(check_if_dirty)
def docker_push(ctx, tag=None):
    if not tag:
        tag = latest_project_version()

    docker_build(ctx, tag=tag)

    assert_tag_not_in_registry(tag)

    push_command = 'docker push {docker_registry_project_url}:{latest_git_tag}'.format(
        docker_registry_project_url=DOCKER_REGISTRY_PROJECT_URL,
        latest_git_tag=tag,
    )
    run(push_command)


@task(help={'dont-deploy': 'Не выкатывать релиз в тестинг. Может быть полезно, если в тестинге запущен регресс, а тебе нужно собрать релиз с квикфиксиком.'})
def full_release(ctx, tag=None, dont_deploy=False):
    bump(ctx, 'minor')
    run('git push')
    run('git push --tags')
    release(ctx, tag, dont_deploy)


@task(help={'dont-deploy': 'Не выкатывать релиз в тестинг. Может быть полезно, если в тестинге запущен регресс, а тебе нужно собрать релиз с квикфиксиком.'})
def release(ctx, tag=None, dont_deploy=False):
    """Собирает версию директории и выкатывает её в testing."""
    print('Pulling latest tags')
    run('git fetch --tags')

    if not tag:
        tag = latest_project_version()

    # обновим документацию
    build_docs(ctx)

    # компилируем локализованные переводы для писем
    compile_translations(ctx)

    assert_tag_not_in_registry(tag)

    push_command = 'releaser release --non-interactive -v {latest_git_tag}'.format(
        latest_git_tag=tag,
    )
    if dont_deploy:
        push_command += ' --skip deploy'

    run(push_command, pty=True)


@task()
def deploy(ctx, environment, tag=None):
    "Выкатывает текущую версию (последнюю из ChangeLog) в заданное окружение. Окружение должно быть test, qa или prod. Если выбран prod, релиз покатится и в internal-prod и в external-prod."
    if not tag:
        tag = latest_project_version()

    envs_map = {
        'prod': ['internal-prod', 'external-prod'],
        'test': ['internal-test'],
        'qa': ['internal-qa'],
        'int-qa': ['integration-qa'],
    }

    if environment not in envs_map:
        click.echo('Поддерживаются только {}'.format(', '.join(envs_map)))

    envs = envs_map[environment]
    for env in envs:
        command = 'releaser deploy -e "{env}" -v "{tag}"'.format(
            env=env,
            tag=tag,
        )
        run(command, pty=True)


@task(check_if_dirty)
def docker_build(ctx, tag=None):
    if not tag:
        tag = latest_project_version()

    assert_tag_not_in_registry(tag)

    # обновим документацию
    build_docs(ctx)

    # компилируем локализованные переводы для писем
    compile_translations(ctx)

    build_command = 'docker build -t {docker_registry_project_url}:{latest_git_tag} -f docker/Dockerfile .'.format(
        docker_registry_project_url=DOCKER_REGISTRY_PROJECT_URL,
        latest_git_tag=tag,
    )
    run(build_command)


@task
def bump(ctx, part):
    run('bumpversion ' + part)


@task()
def run_checks(ctx):
    run("bandit -r ./src -t B102,B301,B302,B307,B313,B314,B315,B316,B317,B318,B319,B320,B403,B506,B602,B604,B605 -lll -iii")


def latest_project_version(silent=False):
    tag = run('git describe --abbrev=0 --tags', hide='out').stdout.strip()
    print(('Latest tag: "%s"' % tag))
    return tag


def read_docker_token():
    """Читает токен из .token или рекомендует получить его.
    """
    try:
        data = json.load(open(os.path.expanduser('~/.docker/config.json')))
        if data.get('credsStore') == 'osxkeychain':
            token = os.environ.get('DOCKER_TOKEN')
            if token:
                return 'OAuth ' + token
            else:
                raise RuntimeError("""Зайдите на:

https://oauth.yandex-team.ru/authorize?response_type=token&client_id=92908ff58d724b0d8479379a65a425d7

и поместите токен в переменную окружения DOCKER_TOKEN.""")

        return 'Basic ' + data['auths']['registry.yandex.net']['auth']
    except:
        raise RuntimeError("""Пожалуйста, залогиньтесь в registry.yandex.net с помощью команды:
    docker login registry.yandex.net""")


def assert_tag_not_in_registry(tag):
    print('Checking docker registry tags...')

    url = 'https://registry.yandex.net/v2/workspace/directory/manifests/{tag}'.format(
        tag=tag,
    )
    headers = {'Authorization': read_docker_token()}
    response = requests.get(url, headers=headers)
    assert response.status_code in (200, 404)
    # если статус 404, значит тега в registry нет
    return response.status_code == 404


@task()
def inner(ctx):
    run('./src/manage.py migrate')
    # Команда, которую нужно будет запустить для заполнения тестовых данных
    # TODO: https://st.yandex-team.ru/DIR-3198
    # run('./src/manage.py fill-dev-database')


@task
def update_requirements(ctx):
    """
    Компилируем зависимости requirements.in в requirements.txt используя pip-compile.
    """
    common_ur(ctx)


@task
def build_dev_container(ctx):
    """
    Собираем образ для разработческого контейнера в Qloud
    """
    url = '{}/{}'.format(
        DOCKER_REGISTRY,
        'workspace/directory-qloud-dev'
    )
    run('docker build -t {url} -f docker/Dockerfile-qloud .'.format(url=url))
    run('docker push {url}'.format(url=url))


@task(help={'days': "Старше скольки дней удалять (по умолчанию 28)", 'last': "Сколько последних тэгов оставить (по умолчанию 15)"}, )
def docker_delete_tags(ctx, days=None, last=None, verbose=False):
    """
    Удаляет старые тэги из registry.yandex.net
    """
    days = days or 28
    last = last or 15

    try:
        days = int(days)
    except ValueError:
        raise ValueError('days must be integer')
    try:
        last = int(last)
    except ValueError:
        raise ValueError('last must be integer')

    headers = {'Authorization': read_docker_token()}

    manifest_headers = headers.copy()
    manifest_headers['Accept'] = 'application/vnd.docker.distribution.manifest.v2+json'

    tags_list_url = 'https://{}/v2/{}/tags/list'.format(DOCKER_REGISTRY, DOCKER_REGISTRY_PROJECT)
    tags_list = requests.get(tags_list_url, headers=headers).json()
    tags = tags_list['tags']
    print('Всего тегов {}'.format(len(tags)))

    ordered_by_date_tags = []
    for tag in tags:
        url = 'https://registry.yandex.net/v2/{repo}/manifests/{tag}'.format(
            repo=DOCKER_REGISTRY_PROJECT,
            tag=tag,
        )
        response = requests.get(url, headers=headers)
        manifest = response.json()
        v1Compatibility = json.loads(manifest['history'][0]['v1Compatibility'])
        created = v1Compatibility['created']
        created = parse(created)
        ordered_by_date_tags.append((created, tag))

    ordered_by_date_tags.sort(key=itemgetter(0), reverse=True)
    if verbose:
        print('\n'.join(map(itemgetter(1), ordered_by_date_tags)))

    utcnow = datetime.now(tz=pytz.UTC)
    deleted = 0
    for created, tag in ordered_by_date_tags[last:]:
        days_diff = (utcnow - created).days
        if days_diff > days:
            url = 'https://registry.yandex.net/v2/{repo}/manifests/{tag}'.format(
                repo=DOCKER_REGISTRY_PROJECT,
                tag=tag,
            )
            response = requests.get(url, headers=manifest_headers)
            digest = hashlib.sha256(response.content).hexdigest()

            delete_url = 'https://registry.yandex.net/v2/{repo}/manifests/{tag}'.format(
                repo=DOCKER_REGISTRY_PROJECT,
                tag='sha256:{}'.format(digest),
            )
            response = requests.delete(delete_url, headers=manifest_headers)
            if response.status_code == 202:
                if verbose:
                    print('Тэг "{}" удален'.format(tag))
                deleted += 1
            else:
                print('Ошибка при удалении тега "{}" ({})'.format(tag, response.status_code))
    print('Удалено {} тегов'.format(deleted))


def get_current_branch():
    response = run('git rev-parse --abbrev-ref HEAD', hide=True)
    return response.stdout.strip()

@task
def clear_merged_branches(ctx):
    """
    Подчищает локальные ветки, которые уже смерджены, а так же удаляет смердженные ветки на GitHub.
    """
    branch = get_current_branch()

    if branch == 'master':
        click.echo('Удаляем локальные бранчи')
        run('git branch --merged | grep -v \'^\*\' | grep -v master | xargs git branch -d')

        click.echo('Удаляем remote бранчи')
        run('for branch in `git branch -r --merged | sed \'s|origin/||\'`;do git push origin --delete "$branch";done')

        click.echo('Подчищаем .git')
        run('git remote prune origin')
        run('git gc')
    else:
        click.echo('Команду надо выполнять на ветке master.')


# добавляем алиасики типа `invoke docker-push`
make_dashed_aliases(list(locals().values()))
