# encoding: utf-8
from __future__ import unicode_literals

from django.db.models import F, Q
from django.utils import timezone

from intranet.yasanta.backend.gifts.models import SantaEntry, Event, Report


def get_event(event_code):
    try:
        return Event.objects.get(code=event_code)
    except Event.DoesNotExist:
        return None


class UnluckyPerson(Exception):
    pass


class SantaEntryCtl(object):

    def __init__(self, event, lucky_login=None):
        self.event = event
        self.lucky_login = lucky_login

    @property
    def santa_entry(self):
        try:
            result = (
                SantaEntry.objects
                .filter(event=self.event)
                .values(
                    'lucky_login',
                    'himself',
                    'collector_login',
                    'collected_count',
                )
                .get(lucky_login=self.lucky_login)
            )

            result['filter_url'] = ' '.join(
                SantaEntry.objects
                .filter(
                    event=self.event,
                    collector_login=self.lucky_login,
                )
                .exclude(lucky_login=self.lucky_login)
                .order_by('lucky_login')
                .values_list('lucky_login', flat=True)
            )

        except SantaEntry.DoesNotExist:
            if self.event.for_everyone:
                entry = self.default_entry
                result = {
                    'lucky_login': entry.lucky_login,
                    'himself': entry.himself,
                    'collector_login': entry.collector_login,
                    'collected_count': entry.collected_count,
                    'filter_url': ''
                }
                return result
            raise UnluckyPerson

        return result

    @property
    def default_entry(self):
        return SantaEntry(
            event=self.event,
            lucky_login=self.lucky_login,
            himself=False,
            collector_login=self.lucky_login,
            collected_count=0,
        )

    def as_dict(self):
        result = dict(self.santa_entry)
        result['collected_for'] = []
        result['collector_for'] = {}

        qs = (
            SantaEntry.objects
            .filter(event=self.event)
            .values_list('lucky_login', 'collected_count', 'instruction')
            .filter(
                Q(collector_login=self.lucky_login)
                | Q(lucky_login=self.lucky_login)
            )
        )

        for lucky_login, collected_count, instruction in qs:
            if collected_count > 0:
                result['collected_for'].append(lucky_login)
            else:
                result['collector_for'][lucky_login] = instruction

        return result

    def unset_collector(self):
        return (
            SantaEntry.objects
            .filter(
                event=self.event,
                collector_login=self.lucky_login,
                collected_count=0,
            )
            .update(collector_login=F('lucky_login'))
        )

    def set_collector(self, lucky_logins):
        return (
            SantaEntry.objects
            .filter(
                event=self.event,
                lucky_login__in=lucky_logins,
                collected_count=0,
                himself=False,
            )
            .exclude(lucky_login=self.lucky_login)
            .update(collector_login=self.lucky_login)
        )

    def set_himself(self, himself):
        params = {'himself': himself}
        if himself:
            params['collector_login'] = self.lucky_login

        return (
            SantaEntry.objects
            .filter(
                event=self.event,
                lucky_login=self.lucky_login,
                collected_count=0,
            )
            .update(**params)
        )

    def collect(self, count, gift_type):
        if self.event.for_everyone:
            exist = (
                SantaEntry.objects
                .filter(
                    event=self.event,
                    lucky_login=self.lucky_login,
                )
                .exists()
            )
            if not exist:
                self.default_entry.save()
        now = timezone.now()
        entry_filter = {
            'event': self.event,
        }
        if self.event.max_count != 0:
            entry_filter['collected_count__lte'] = self.event.max_count - count

        (
            SantaEntry.objects
            .filter(
                **entry_filter
            )
            .filter(
                Q(collector_login=self.lucky_login)
                | Q(lucky_login=self.lucky_login)
            )
            .update(
                collected_count=F('collected_count') + count,
                collected_time=now,
                collector_login=self.lucky_login,
                collected_type=gift_type,
            )
        )

        entry_ids_for_report = list(
            SantaEntry.objects
            .filter(collected_time=now)
            .values_list('id', flat=True)
        )
        Report.objects.bulk_create(
            [
                Report(
                    entry_id=e,
                    time=now,
                    action='collect',
                    count=count,
                )
                for e in entry_ids_for_report
            ]
        )

        # TODO тут можно отправлять письмо автору

        return list(
            SantaEntry.objects
            .filter(
                event=self.event,
                collector_login=self.lucky_login,
                collected_time=now,
            )
            .values_list('lucky_login', flat=True)
        )

    def collect_back(self, count):
        now = timezone.now()

        try:
            entry = (
                SantaEntry.objects
                .get(
                    event=self.event,
                    lucky_login=self.lucky_login,
                    collected_count__gte=count,
                )
            )
        except SantaEntry.DoesNotExist:
            raise UnluckyPerson

        entry.collected_count = F('collected_count') - count
        entry.collected_time = now
        entry.save()

        Report.objects.create(
            entry=entry,
            time=now,
            action='return',
            count=count,
        )

        return self.lucky_login

    def report(self):
        entries = (
            SantaEntry.objects
            .filter(event=self.event)
            .order_by('-collected_count', '-collected_time')
            .values(
                'lucky_login',
                'collector_login',
                'collected_time',
                'instruction',
                'collected_type_id',
                'collected_type__name',
                'collected_type__name_en',
                'collected_count',
            )
        )

        result = {
            'entries': [
                {
                    'lucky_login': e['lucky_login'],
                    'collector_login': e['collector_login'] if e['collected_time'] else '',
                    'collected_time': e['collected_time'] or '',
                    'instruction': e['instruction'],
                    'gift_type_id': e['collected_type_id'] or '',
                    'gift_type_name': e['collected_type__name'] or '',
                    'collected_count': e['collected_count'],
                }
                for e in entries
            ],
            'all_count': len(entries),
            'collected_count': entries.filter(collected_count__gt=0).count(),
            'event': event_as_dict(self.event),
        }

        return result


def event_as_dict(event):
    return {
        'id': event.id,
        'code': event.code,
        'name': event.name,
        'comment': event.comment,
        'admins': event.admins.split(',') if event.admins else [],
        'gift_types': list(event.gifttype_set.values('id', 'name', 'name_en')),
        'max_count': event.max_count,
        'returnable': event.returnable,
        'for_everyone': event.for_everyone,
    }
