# coding: utf-8
from __future__ import unicode_literals

from django.db import models


class Event(models.Model):
    class Meta:
        db_table = 'yasanta_event'

    code = models.CharField(
        max_length=32,
        verbose_name='Текстовый идентификатор',
        unique=True,
    )
    name = models.CharField(
        max_length=256,
        verbose_name='Название события',
    )
    comment = models.CharField(
        max_length=256,
        verbose_name='Комментарий',
    )
    admins = models.CharField(
        max_length=1024,
        verbose_name='Логины админов',
    )
    returnable = models.BooleanField(
        default=False,
        verbose_name='Можно вернуть подарок',
    )
    max_count = models.IntegerField(
        default=1,
        verbose_name='Максимальное количество (0 - безлимит)',
    )
    for_everyone = models.BooleanField(
        default=False,
        verbose_name='Подарок доступен любому человеку'
    )


class GiftType(models.Model):
    class Meta:
        ordering = ['position', 'id']

    event = models.ForeignKey(
        to=Event,
        verbose_name='Событие',
        on_delete=models.CASCADE,
    )
    name = models.CharField(
        max_length=256,
        verbose_name='Название подарка',
    )
    name_en = models.CharField(
        max_length=256,
        verbose_name='Название на английском',
    )
    position = models.IntegerField(
        default=0,
        verbose_name='Позиция в выдаче',
    )


class SantaEntry(models.Model):
    class Meta:
        db_table = 'yasanta_santaentry'
        unique_together = (('event', 'lucky_login'),)

    event = models.ForeignKey(
        to=Event,
        verbose_name='Событие',
        on_delete=models.CASCADE,
    )
    lucky_login = models.CharField(
        max_length=256,
        verbose_name='Логин сотрудника',
    )
    himself = models.BooleanField(
        default=False,
        verbose_name='Сотрудник сам заберет подарок',
    )
    collector_login = models.CharField(
        db_index=True,
        blank=False,
        null=False,
        max_length=256,
        verbose_name='Логин забравшего сотрудника',
    )
    collected_time = models.DateTimeField(
        null=True,
        blank=True,
        verbose_name='Момент, когда забрали',
    )
    collected_count = models.IntegerField(
        default=0,
        verbose_name='Количество, которое забрал',
    )
    filter_url = models.TextField(
        null=False,
        blank=True,
        default='',
        verbose_name='Список логинов, для кого хочется забрать',
    )
    instruction = models.CharField(
        blank=True,
        null=False,
        max_length=256,
        default='',
        verbose_name='Инструкция к подарку',
    )
    collected_type = models.ForeignKey(
        to=GiftType,
        verbose_name='Тип подарка',
        on_delete=models.SET_NULL,
        null=True,
        default=None,
        blank=True,
    )


class Report(models.Model):
    ACTIONS = (
        ('collect', 'Взял'),
        ('return', 'Вернул'),
    )

    entry = models.ForeignKey(
        to=SantaEntry,
        on_delete=models.CASCADE,
    )
    time = models.DateTimeField(
        verbose_name='Время',
    )
    action = models.CharField(
        max_length=32,
        choices=ACTIONS,
        verbose_name='Что сделал'
    )
    count = models.IntegerField(
        default=1,
        verbose_name='Количество'
    )
