from django.http import HttpResponse

from intranet.yasanta.backend.utils import responding_json, responding_xlsx
from intranet.yasanta.backend.staff import get_key_owner, KeyNotFound

from intranet.yasanta.backend.gifts.controller import SantaEntryCtl, UnluckyPerson, get_event, event_as_dict
from intranet.yasanta.backend.gifts.forms import SantaEntryForm, CollectForm


@responding_json
def auth(request, event_code):
    event = get_event(event_code=event_code)
    if event is None:
        return {'error': 'unknown_event'}, 404

    if request.yauser.login not in event.admins:
        return {'error': 'you_are_not_admin'}, 403

    key = request.GET.get('key')

    try:
        key_data = get_key_owner(key)
    except KeyNotFound:
        return {'error': 'bad_key'}, 400

    if not key_data['login']:
        return {'error': 'unknown_person'}, 400

    if not key_data['status']:
        return {'error': 'locked_key'}, 400

    ctl = SantaEntryCtl(event=event, lucky_login=key_data['login'])
    try:
        return ctl.as_dict()
    except UnluckyPerson:
        return {'error': 'unlucky'}, 404


@responding_json
def prepare(request, event_code, person_login=None):
    event = get_event(event_code=event_code)
    if event is None:
        return {'error': 'unknown_event'}, 404
    if person_login is None:
        person_login = request.yauser.login
    elif person_login != request.yauser.login and request.yauser.login not in event.admins:
        return {'error': 'you_are_not_admin'}, 403

    ctl = SantaEntryCtl(event=event, lucky_login=person_login)
    try:
        santa_entry = ctl.santa_entry
    except UnluckyPerson:
        return {'error': 'unlucky'}, 404

    data = request.POST if request.method == 'POST' else None
    santa_entry_form = SantaEntryForm(initial=santa_entry, data=data)

    if request.method == 'POST':
        if not santa_entry_form.is_valid():
            return santa_entry_form.errors, 400

        if santa_entry['collected_count'] == 0:
            ctl.set_himself(santa_entry_form.cleaned_data['himself'])

        ctl.unset_collector()

        logins = santa_entry_form.cleaned_data['filter_url']
        if logins:
            ctl.set_collector(logins)

        santa_entry_form = SantaEntryForm(initial=ctl.santa_entry)

    result = ctl.as_dict()
    result.update(santa_entry_form.as_dict())

    return result


@responding_json
def collect(request, event_code, person_login):
    event = get_event(event_code=event_code)
    if event is None:
        return {'error': 'unknown_event'}, 404

    if request.yauser.login not in event.admins:
        return {'error': 'you_are_not_admin'}, 403

    form = CollectForm(data=request.POST, initial={'event': event})

    if not form.is_valid():
        return form.errors, 400

    ctl = SantaEntryCtl(event=event, lucky_login=person_login)
    try:
        return {'collect_for_logins': ctl.collect(**form.cleaned_data)}
    except UnluckyPerson:
        return {'error': 'unlucky'}, 404


@responding_json
def collect_back(request, event_code, person_login):
    event = get_event(event_code=event_code)
    if event is None:
        return {'error': 'unknown_event'}, 404

    if request.yauser.login not in event.admins:
        return {'error': 'you_are_not_admin'}, 403

    if not event.returnable:
        return {'error': 'gift_cannot_be_returned'}, 400

    count = int(request.POST.get('count', 1))

    ctl = SantaEntryCtl(event=event, lucky_login=person_login)
    try:
        return {'return_for_login': ctl.collect_back(count)}
    except UnluckyPerson:
        return {'error': 'unlucky'}, 404


def report_data(user_login, event_code):
    event = get_event(event_code=event_code)
    if event is None:
        return {'error': 'unknown_event'}, 404

    if user_login not in event.admins:
        return {'error': 'you_are_not_admin'}, 403

    ctl = SantaEntryCtl(event=event)

    return ctl.report()


@responding_json
def report(request, event_code):
    return report_data(request.yauser.login, event_code)


@responding_xlsx('gifts_report')
def export_excel(request, event_code):
    data = report_data(request.yauser.login, event_code)

    if 'error' in data:
        return HttpResponse(data)

    page = [[
        'recipient',
        'collected',
        'collector',
        'time',
        'instruction',
        'type',
    ]]

    page.extend([[
        d['lucky_login'],
        d['collected_count'],
        d['collector_login'],
        d['collected_time'] and d['collected_time'].isoformat()[:19],
        d['instruction'],
        d['gift_type_name'],
    ] for d in data['entries']])

    return [['report', page]]


@responding_json
def info(request, event_code):
    event = get_event(event_code=event_code)
    if event is None:
        return {'error': 'unknown_event'}, 404
    return event_as_dict(event)
