# encoding: utf-8
from __future__ import unicode_literals

from json import JSONEncoder
from json import dumps

import io
from datetime import date, datetime, time
from decimal import Decimal
import re
from functools import partial, wraps
import xlsxwriter

from django.utils.functional import Promise
from django.db.models import Model
from django.http import HttpResponse
from django.utils.encoding import smart_text, force_text


class CustomJSONEncoder(JSONEncoder):
    def default(self, o):
        if isinstance(o, (datetime, time, date)):
            return re.sub(r'\.\d{6}', '', o.isoformat())
        elif isinstance(o, set):
            return list(o)
        elif isinstance(o, Promise):
            return force_text(o)
        elif isinstance(o, Model):
            return str(o.pk)
        elif isinstance(o, Decimal):
            return str(o)
        else:
            return super(CustomJSONEncoder, self).default(o)


custom_dumps = partial(dumps, cls=CustomJSONEncoder)


def make_json_response(request, data, status=200, dumps=custom_dumps):
    if isinstance(data, (list, dict)):
        data = dumps(data)

    data = smart_text(data)

    return HttpResponse(
        data,
        content_type='application/json; charset=utf-8',
        status=status,
    )


def responding_json(view, dumps=custom_dumps):
    """
    Декоратор для обертки респонза в JSON
    @return: HttpResponse
    """
    @wraps(view)
    def wrapper(request, *args, **kwargs):
        data = view(request, *args, **kwargs)

        if isinstance(data, HttpResponse):
            return data

        status = 200
        if isinstance(data, tuple):
            data, status = data

        return make_json_response(request, data, status)

    return wrapper


def make_xlsx_response(file_name, data, status=200):
    stream = io.BytesIO()
    with xlsxwriter.Workbook(stream) as wb:
        for sheet_name, sheet_data in data:
            ws = wb.add_worksheet(name=sheet_name)
            for i, row in enumerate(sheet_data):
                for j, cell_data in enumerate(row):
                    ws.write(i, j, cell_data)

    data = stream.getvalue()
    response = HttpResponse(
        data,
        content_type=(
            'application/vnd'
            '.openxmlformats-officedocument'
            '.spreadsheetml.sheet; charset=utf-8'
        ),
        status=status,
    )

    today = date.today().isoformat()
    response['Content-Disposition'] = (
        'attachment; filename="%s_%s.xlsx"' % (file_name, today)
    )
    response['Content-Length'] = len(data)
    return response


def responding_xlsx(file_name=None):
    """
    Декоратор для обертки респонза в xls
    @return: HttpResponse
    """

    def decorator(view):

        @wraps(view)
        def wrapper(request, *args, **kwargs):
            data = view(request, *args, **kwargs)

            if isinstance(data, HttpResponse):
                return data

            status = 200

            if isinstance(data, tuple):
                data, status = data

            return make_xlsx_response(file_name, data, status)

        return wrapper

    return decorator
