'use strict';

var fs = require('fs');

var techs = {
    fileProvider: require('enb/techs/file-provider'),
    fileMerge: require('enb/techs/file-merge'),
    fileCopy: require('enb/techs/file-copy'),
    levels: require('enb-bem-techs/techs/levels'),
    deps: require('enb-bem-techs/techs/deps'),
    files: require('enb-bem-techs/techs/files'),
    stylus: require('enb-stylus/techs/stylus'),
    browserify: require('./techs/browserify'),
    bemhtml: require('enb-bemxjst/techs/bemhtml'),
    html: require('./techs/bemjson-to-lang-html'),
    bemjsonToBemdecl: require('enb-bem-techs/techs/bemjson-to-bemdecl'),
    i18n: require('enb-bem-i18n/techs/i18n'),
    keysets: require('enb-bem-i18n/techs/keysets'),
    borschik: require('enb-borschik/techs/borschik')
};

function getLevels(config) {
    return [
        {path: 'node_modules/islands/common.blocks', check: false},
        {path: 'node_modules/islands/desktop.blocks', check: false},
        {path: 'node_modules/meccano/desktop.blocks', check: false},
        {path: 'node_modules/awesome-icon/desktop.blocks', check: false},
        {path: 'node_modules/s-form/blocks/desktop', check: false},
        {path: 'blocks/external', check: true},
        {path: 'blocks/desktop', check: true}
    ].map(function(l) {
        return config.resolvePath(l);
    });
}

/**
 * Возвращает объект-технологию для `nodeConfig`
 *
 * @param {String} tech название технологии
 * @param {Object} [params] параметры для технологии
 * @returns {*[]}
 */
function use(tech, params) {
    return [
        techs[tech],
        params || {}
    ];
}

module.exports = function(config) {
    config.setLanguages(['ru', 'en']);

    var production = process.argv.indexOf('--production') !== -1;

    var bundles = fs.readdirSync('./bundles').filter(function(name) {
        return name !== '_';
    });

    /* common */
    bundles.forEach(function(name) {
        config.node('bundles/' + name, function(nodeConfig) {
            nodeConfig.addTechs([
                use('levels', {
                    target: '?.levels',
                    levels: getLevels(config)
                }),
                use('fileProvider', {
                    target: '?.bemjson.js'
                }),
                use('bemjsonToBemdecl'),
                use('deps'),
                use('files'),
                use('keysets', {lang: '{lang}'}),
                use('i18n', {
                    lang: '{lang}'
                }),
                use('bemhtml'),
                use('browserify', {
                    target: '?.browser.js',
                    opts: {debug: !production}
                }),
                use('stylus', {
                    target: '?.styled.css',
                    autoprefixer: {
                        browsers: [
                            "last 2 Chrome version",
                            "last 2 Firefox version",
                            "last 2 Safari version",
                            "last 2 Opera version"
                        ]
                    }
                }),
                use('borschik', {
                    sourceTarget: '?.browser.js',
                    destTarget: '?.js',
                    minify: production
                }),
                use('borschik', {
                    sourceTarget: '?.styled.css',
                    destTarget: '?.css',
                    minify: production,
                    freeze: true
                }),
                use('html', {
                    langFile: '?.lang.ru.js',
                    target: '?.html',
                    globals: require('./globals').globals
                })
            ]);

            nodeConfig.addTargets([
                '?.css',
                '?.js',
                '?.lang.{lang}.js',
                '?.bemhtml.js',
                '?.html'
            ]);
        });
    });

    /* tests */
    config.includeConfig('enb-bem-tmpl-specs');
    config.module('enb-bem-tmpl-specs')
        .createConfigurator('test:templates', {
            coverage: {
                engines: ['bemhtml'],
                reportDirectory: 'tests/coverage/template',
                exclude: ['**/node_modules/**'],
                reporters: ['lcov']
            }
        })
        .configure({
            prependFiles: [
                require.resolve('./globals')
            ],
            langs: true,
            saveHtml: true,
            destPath: 'tests/templates',
            levels: ['blocks/desktop'],
            sourceLevels: getLevels(config),
            engines: {
                'bemhtml': {
                    tech: 'enb-bemxjst/techs/bemhtml'
                }
            }
        });
};
