var fs = require('fs');
var vm = require('vm');
var enb = require('enb');
var buildFlow = enb.buildFlow || require('enb/lib/build-flow');
var clearRequire = require('clear-require');
var _ = require('lodash');

/**
 * @class BemjsonToHtmlTech
 * @augments {BaseTech}
 * @classdesc
 *
 * Build HTML file.<br/><br/>
 *
 * This tech uses `BEMHTML.apply(bemjson)` to build HTML.
 *
 * @param {Object}  [options]                            Options
 * @param {String}  [options.target='?.html']            Path to a target with HTML file.
 * @param {String}  [options.bemhtmlFile='?.bemhtml.js'] Path to a file with compiled BEMHTML module.
 * @param {String}  [options.bemjsonFile='?.bemjson.js'] Path to BEMJSON file.
 *
 * @example
 * var BemjsonToHtmlTech = require('enb-bemxjst/techs/bemjson-to-html'),
 *     BemhtmlTech = require('enb-bemxjst/techs/bemhtml'),
 *     FileProvideTech = require('enb/techs/file-provider'),
 *     bem = require('enb-bem-techs');
 *
 * module.exports = function(config) {
 *     config.node('bundle', function(node) {
 *         // get BEMJSON file
 *         node.addTech([FileProvideTech, { target: '?.bemjson.js' }]);
 *
 *         // get FileList
 *         node.addTechs([
 *             [bem.levels, levels: ['blocks']],
 *             bem.bemjsonToBemdecl,
 *             bem.deps,
 *             bem.files
 *         ]);
 *
 *         // build BEMHTML file
 *         node.addTech(BemhtmlTech);
 *         node.addTarget('?.bemhtml.js');
 *
 *         // build HTML file
 *         node.addTech(BemjsonToHtmlTech);
 *         node.addTarget('?.html');
 *     });
 * };
 */
module.exports = buildFlow.create()
    .name('bemjson-to-lang-html')
    .target('target', '?.{lang}.html')
    .defineOption('globals', {})
    .useSourceFilename('bemhtmlFile', '?.bemhtml.js')
    .useSourceFilename('bemjsonFile', '?.bemjson.js')
    .useSourceFilename('langFile', '?.lang.{lang}.js')
    .builder(function(bemhtmlFilename, bemjsonFilename, langFilename) {
        var _this = this;

        clearRequire(bemjsonFilename);
        clearRequire(langFilename);

        var bemhtml = _.extend({
            console: console,
            BEM: {I18N: require(langFilename)}
        }, _this._globals);

        bemhtml.global = bemhtml;

        vm.runInNewContext(fs.readFileSync(bemhtmlFilename, 'utf8'), bemhtml, {
            filename: bemhtmlFilename,
            displayErrors: true
        });

        return _this.render(bemhtml, require(bemjsonFilename));
    })
    .methods({
        render: function(bemhtml, bemjson) {
            return bemhtml.BEMHTML.apply(bemjson);
        }
    })
    .createTech();
