'use strict';

var Promise = require('bluebird');
/* настройка bluebird */
Promise.config({
    cancellation: true,
    warnings: false
});

$(function() {
    $.extend(BEM.prototype, {
        /**
         * Отменить все текущие запросы
         */
        abortAllRequests: function() {
            if (this._requestsQueue) {
                this._requestsQueue._queue.forEach(function(reqest) {
                    reqest.abort();
                });
            }
        },
        /**
         * Отменить запрос по имени
         * @param {String} name
         */
        abortRequest: function(name) {
            if (this._requestsStorage && this._requestsStorage[name]) {
                this._requestsStorage[name].abort();
            }
        },
        /**
         * Выполняет асинхронный запрос
         * Отменяет предыдущий запрос с тем же id
         * @param {Object} options
         * @param {String} [id] — идентификатор запроса (в контексте блока)
         * @returns {Promise}
         */
        request: function(options, id) {
            var storage = this._requestsStorage || (this._requestsStorage = {});
            var queue = this._requestsQueue || (this._requestsQueue = {
                _queue: [],
                enqueue: function(request) {
                    this._queue.push(request);
                },
                dequeue: function(request) {
                    this._queue = this._queue.filter(function(item) {
                        return item !== request;
                    });
                }
            });

            /* Отмена предыдущего запроса */
            if (id && storage[id]) {
                storage[id].abort();
            }

            return new Promise(function(resolve, reject, cancel) {
                var request = $.ajax(options);

                queue.enqueue(request);

                /* Сохранение текущего запроса */
                if (id) {
                    storage[id] = request;
                }

                request.then(function(data) {
                    if (id) {
                        delete storage[id];
                    }

                    queue.dequeue(request);

                    resolve(data);
                }, function(response, status, message) {
                    var error = {
                        code: response.status,
                        status: status,
                        message: message
                    };

                    try {
                        _.assign(error, JSON.parse(response.responseText));
                    } catch (e) {
                        // noop
                    }

                    reject(error);
                });

                cancel(function() {
                    request.abort();
                });
            }).bind(this);
        }
    });
});
