import {
    FooterData,
    Menu,
    MenuHeaderItem,
    ReportData
} from 'client/bundles/types';

declare global {
    type BunkerLang = 'en' | 'ru';
    type BunkerPageName = keyof BunkerPages;
    type BunkerPageSection =
        | 'quality-of-life'
        | 'investing-in-people'
        | 'environmental-impact'
        | 'ethics-integrity';

    type Bunker = {
        [lang in BunkerLang]: {
            dictionary: BunkerDictionary;
            common: BunkerCommon;
            pages: BunkerPages;
        };
    };

    interface BunkerDictionary {
        [key: string]: string;
    }

    interface BunkerCommon {
        title: string;
        links: MenuHeaderItem[];
        seo: {
            description: string;
            ogimage: string;
        };
        yandexMainPage: string;
        agenda: MenuHeaderItem;
        report: ReportData;
        menu: Menu;
        footer: FooterData;
    }

    interface BunkerCommonData {
        dictionary: BunkerDictionary;
        common: BunkerCommon;
    }

    interface BunkerPageBase {
        category?: string;
        title: string;
        description: string;
        sections?: Record<string, Record<string, unknown>>;
        streams?: any;
        themes?: any;
        summary?: any;
        governs?: any;
        link?: string;
        menu: MenuItem[];
    }

    interface TextItem {
        title: string;
        text: string;
        tooltip?: string;
    }

    interface PopupTextItem {
        title: string;
        text: string;
        image?: string;
        popup?: string;
    }

    interface CardTabsItem {
        isActive?: boolean;
        title: string;
        period?: string;
        text: string;
        image?: string;
        bio?: string;
        photo?: string;
        facts?: TextItem[];
        titleUrl?: string;
        url?: string;
    }

    interface AreasItem {
        title: string;
        facts: TextItem[];
        projects: {
            title: string;
            url?: string;
            text: string;
        }[];
    }

    interface Docs {
        id: string;
        title?: string;
        items: DocItem[];
    }

    interface DocItem {
        name: string;
        url: string;
        format?: string;
        size?: string;
    }

    interface BunkerPageAnticorruption
        extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            statement: {
                id: string;
                title: string;
                items: CardTabsItem[];
            };
            docs: Docs;
        };
    }

    interface BunkerPageAccessible extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            hand: {
                id: string;
                title: string;
                url: string;
                description: string;
                items: {
                    left: TextItem[];
                    right: TextItem[];
                };
            };
            dyk: {
                id: string;
                title: string;
                description: string;
            };
            docs: Docs;
        };
    }

    interface FaqItem {
        id: string;
        title: string;
        topics: TextItem[];
    }

    interface BunkerPageFaq extends Omit<BunkerPageBase, 'sections'> {
        themes: {
            items: FaqItem[];
        };
    }

    interface DownloadYear {
        year: string;
        docs: Docs;
    }

    interface BunkerPageDownload extends Omit<BunkerPageBase, 'sections'> {
        years: DownloadYear[];
    }

    interface BunkerPageDownloadWebcast
        extends Omit<BunkerPageBase, 'sections'> {
        link: string;
    }

    interface BunkerPageAgenda extends Omit<BunkerPageBase, 'sections'> {
        streams: {
            dictionary: StreamDictionary;
            contributions: StreamDictionary;
            items: StreamItem[];
        };
        summary: {
            title: string;
            description?: string;
            items: TextItem[];
        };
    }

    interface CardDigitData {
        pre?: string;
        num: string;
        text?: string;
        tooltip?: string;
    }

    interface PracticumFactsItem {
        title: string;
        tooltip?: string;
    }

    interface PracticumFacts {
        first: {
            facts: string[];
            digit: CardDigitData;
        };
        second: {
            facts: PracticumFactsItem[];
        };
    }

    interface BunkerPageEducation extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            initiatives: {
                id: string;
                title: string;
                segalovich: CardTabsItem;
                items: CardTabsItem[];
                reports: {
                    title: string;
                    items: DocItem[];
                };
            };
            areas: {
                id: string;
                title: string;
                items: AreasItem[];
            };
            practicum: {
                id: string;
                title: string;
                description?: string;
                data: PracticumFacts;
            };
        };
    }

    interface BunkerPageConvenience extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            facts: {
                id: string;
                title: string;
                items: {
                    icon: 'plus' | 'yandex' | 'eats' | 'taxi';
                    title: string;
                    text: string;
                }[];
            };
            support: {
                id: string;
                title: string;
                description?: string;
                items: CardTabsItem[];
            };
            docs: Docs;
        };
    }

    interface BunkerPagePartners extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            partners: {
                id: string;
                items: TextItem[];
                others: string;
            };
            principles: {
                id: string;
                title: string;
                items: TextItem[];
            };
            benefits: {
                id: string;
                title: string;
                items: CardTabsItem[];
            };
            safety: {
                id: string;
                title: string;
                description?: string;
                items: {
                    left: PopupTextItem[];
                    right: PopupTextItem[];
                };
            };
            docs: Docs;
        };
    }

    interface DevelopingSectionItems {
        left: {
            title: string;
            text: string;
            image: string;
            popup?: string;
        };
        right: {
            title: string;
            text: string;
            services: DevelopingServicesGroup[];
            popup?: string;
        };
    }

    interface DevelopingServicesItem {
        title: string;
        icon?: string;
    }

    interface DevelopingServicesGroup {
        title: string;
        services: DevelopingServicesItem[];
    }

    interface WellbeingComfortFact {
        digit: CardDigitData;
        image: string;
    }

    interface ChartItemImage {
        url: string;
        urltumbler?: string;
        width?: string;
        height: string;
    }

    interface ChartItem {
        title: string;
        tooltip?: string;
        image: {
            desktop: ChartItemImage;
            mobile: ChartItemImage;
        };
        buttontumbler?: string;
        text?: string;
        texttumbler?: string;
        imagetext: string;
        imagetexttumbler?: string;
        tooltips?: string[];
        tooltipstumbler?: string[];
    }

    interface ChartData {
        title: string;
        tooltip?: string;
        description?: string;
        items: ChartItem[];
    }

    interface WorkforceEngagement {
        description?: string;
        items: CardTabsItem[];
    }

    interface BunkerPageEmployees extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            workforce: {
                id: string;
                title: string;
                team: ChartData;
                women: ChartData;
                fact: {
                    image: string;
                    engagement: WorkforceEngagement;
                };
            };
            diversity: {
                id: string;
                title: string;
                description: string;
                items: {
                    left: PopupTextItem[];
                    right: PopupTextItem[];
                };
            };
            developing: {
                id: string;
                title: string;
                description: string;
                items: DevelopingSectionItems;
            };
            wellbeing: {
                id: string;
                title: string;
                description: string;
                finances: {
                    title: string;
                    items: CardTabsItem[];
                };
                comfort: {
                    id: string;
                    title: string;
                    fact: WellbeingComfortFact;
                    items: CardTabsItem[];
                };
            };
        };
    }

    interface LinkData {
        text: string;
        url: string;
    }

    interface MainServices {
        title: string;
        text: string;
        link: LinkData;
    }

    interface BunkerPageMain extends BunkerPageBase {
        results: {
            year: string;
            title: string;
            items: {
                title: string;
                text: string;
                icon: string;
                url: string;
            }[];
            link: LinkData;
        };
        techs: {
            title: string;
            services: MainServices;
        };
        governs: {
            title: string;
            description: string;
        };
    }

    interface BunkerPageNotFound {
        title: string;
        text: string;
    }

    interface BunkerPageProcurement extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            facts: {
                id: string;
                title: string;
                items: {
                    title: string;
                    text: string;
                }[];
            };
            suppliers: {
                id: string;
                title: string;
                items: {
                    title: string;
                    text: string;
                }[];
            };
            docs: Docs;
        };
    }

    interface BunkerPageQuality extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            search: {
                id: string;
                title: string;
                description?: string;
                items: {
                    title: string;
                    text: string;
                }[];
            };
            advertising: {
                id: string;
                title: string;
                description?: string;
                facts?: TextItem[];
                items: {
                    title: string;
                    text: string;
                }[];
            };
            blogging: {
                id: string;
                title: string;
                description?: string;
                items: {
                    title: string;
                    text: string;
                    facts?: TextItem[];
                }[];
            };
            docs: Docs;
        };
    }

    interface StandartsItem {
        title: string;
        tooltip?: string;
    }

    interface BunkerPageSecurity extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            standarts: {
                id: string;
                title: string;
                items: StandartsItem[];
            };
            privacy: {
                id: string;
                title: string;
                description: string;
                tool: {
                    title: string;
                    description: string;
                    popup: string;
                    image?: string;
                };
                other: {
                    items: {
                        left: TextItem[];
                        right: TextItem[];
                    };
                };
            };
            bug: {
                id: string;
                title: string;
                url: string;
                description: string;
            };
            questions: {
                id: string;
                title: string;
                url?: string;
                description: string;
                items: TextItem[];
            };
            docs: Docs;
        };
    }

    interface DatacenterLocation {
        country: string;
        names: string;
    }

    interface BunkerPageEnergy extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            consumption: {
                id: string;
                title: string;
                pue: ChartData;
                electricity: ChartData;
                fact: {
                    image: string;
                    digit: CardDigitData;
                };
            };
            efficiency: {
                id: string;
                title: string;
                items: {
                    left: {
                        title: string;
                        text: string;
                        image?: string;
                    }[];
                    right: PopupTextItem[];
                };
            };
            datacenters: {
                id: string;
                title: string;
                locations: DatacenterLocation[];
                link: LinkData;
            };
        };
    }

    interface BunkerPageCarbon extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            ghg: {
                id: string;
                title?: string;
                chart: ChartData;
            };
            decarbonizing: {
                id: string;
                title?: string;
                description?: string;
            };
            empowering: {
                id: string;
                title?: string;
                description?: string;
                items: CardTabsItem[];
            };
        };
    }

    interface BunkerPageWaste extends Omit<BunkerPageBase, 'sections'> {
        sections: {
            packaging: {
                id: string;
                title: string;
                market: ChartData;
            };
            facts: {
                id: string;
                title: string;
                items: TextItem[];
            };
            approach: {
                id: string;
                title: string;
                description?: string;
                items: {
                    title: string;
                    text: string;
                    popup?: string;
                    doc?: DocItem;
                }[];
            };
            engaging: {
                id: string;
                title: string;
                items: CardTabsItem[];
            };
            keeping: {
                id: string;
                title: string;
                items: CardTabsItem[];
            };
        };
    }

    interface NotesItem {
        title?: string;
        url?: string;
        text: string;
    }

    interface StreamCard {
        title: string;
        type: string;
        link: string;
        contribution: string[];
        goal: string[];
        metrics: NotesItem[];
    }

    interface StreamItem {
        numberText: string;
        title: string;
        id: string;
        cards: StreamCard[];
    }

    interface StreamDictionary {
        [key: string]: string;
    }

    interface BunkerPages {
        anticorruption: BunkerPageAnticorruption;
        accessible: BunkerPageAccessible;
        agenda: BunkerPageAgenda;
        faq: BunkerPageFaq;
        download: BunkerPageDownload;
        'download[webcast]': BunkerPageDownloadWebcast;
        convenience: BunkerPageConvenience;
        education: BunkerPageEducation;
        partners: BunkerPagePartners;
        employees: BunkerPageEmployees;
        procurement: BunkerPageProcurement;
        quality: BunkerPageQuality;
        security: BunkerPageSecurity;
        carbon: BunkerPageCarbon;
        energy: BunkerPageEnergy;
        waste: BunkerPageWaste;
        main: BunkerPageMain;
        'not-found': BunkerPageNotFound;
    }
}
