import React, { Component } from 'react';

import cn from 'client/utils/cn';
import Card from 'client/components/card';
import HintIcon from 'client/components/hint-icon';
import TabsPanes from 'client/components/tabs-panes';
import Title from 'client/components/title';
import Tumbler from 'client/components/tumbler';
import 'client/components/text/index.css';
import 'client/components/visually-hidden/index.css';

import './index.css';

const b = cn('chart');

interface Props {
    data: ChartData;
}

interface State {
    isTumblerChecked: boolean;
}

export default class Chart extends Component<Props, State> {
    constructor(props: Props) {
        super(props);

        this.state = {
            isTumblerChecked: false
        };
    }

    toggleImage = (checked: boolean): void => {
        this.setState({ isTumblerChecked: checked });
    };

    renderNavItem = (item: ChartItem, i: number): React.ReactNode => {
        return (
            <div
                className={b('nav-item', { hint: Boolean(item.tooltip) })}
                key={`chart-nav-${i}`}
            >
                {item.title}

                {item.tooltip && (
                    <div className={b('nav-hint-icon')}>
                        <HintIcon
                            hint={item.tooltip}
                            mixClassName={b('hint-item')}
                            width={350}
                        />
                    </div>
                )}
            </div>
        );
    };

    renderHint = (tooltip: string, i: number): React.ReactNode => {
        if (!tooltip) {
            return null;
        }

        return (
            <div
                className={b('image-hint-icon', { num: i + 1 })}
                key={`hint-${i}`}
            >
                <HintIcon
                    hint={tooltip}
                    mixClassName={b('image-hint-item')}
                    width={200}
                />
            </div>
        );
    };

    renderTooltips = (tooltips: string[] | undefined): React.ReactNode => {
        if (!tooltips || tooltips.length === 0) {
            return null;
        }

        return tooltips.map(this.renderHint);
    };

    renderImage = (
        data: ChartItemImage,
        item: ChartItem,
        device: string
    ): React.ReactNode => {
        const { title, tooltips, tooltipstumbler } = item;

        return (
            <div
                className={b(`image-${device}`)}
                style={{
                    maxWidth: data.width ? `${data.width}px` : null,
                    maxHeight: `${data.height}px`
                }}
                aria-hidden="true"
            >
                <div className={b('image-box')}>
                    <img className={b('image')} src={data.url} alt={title} />

                    {this.renderTooltips(tooltips)}
                </div>

                {data.urltumbler && (
                    <div className={b('image-box', { tumbler: true })}>
                        <img
                            className={b('image')}
                            src={data.urltumbler}
                            alt={title}
                        />

                        {this.renderTooltips(tooltipstumbler)}
                    </div>
                )}
            </div>
        );
    };

    renderNavContent = (item: ChartItem, i: number): React.ReactNode => {
        const { isTumblerChecked } = this.state;
        const imageDesktop = item.image.desktop;
        const imageMobile = item.image.mobile;
        const hasTumblerData =
            item.buttontumbler &&
            imageDesktop.urltumbler &&
            imageMobile.urltumbler;

        let { text } = item;
        if (hasTumblerData && isTumblerChecked) {
            text = item.texttumbler;
        }

        let { imagetext } = item;
        if (hasTumblerData && isTumblerChecked && item.imagetexttumbler) {
            imagetext = item.imagetexttumbler;
        }

        return (
            <div className={b('content')} key={`chart-content-${i}`}>
                <figure
                    className={b('content-image', {
                        tumbler: hasTumblerData && isTumblerChecked
                    })}
                >
                    {this.renderImage(imageDesktop, item, 'desktop')}

                    {this.renderImage(imageMobile, item, 'mobile')}

                    {imagetext && (
                        <figcaption className="visually-hidden">
                            {imagetext}
                        </figcaption>
                    )}
                </figure>

                {hasTumblerData && (
                    <Tumbler
                        text={item.buttontumbler}
                        onChange={this.toggleImage}
                    />
                )}

                {text && <div className={b('content-text')}>{text}</div>}
            </div>
        );
    };

    render(): React.ReactNode {
        const {
            data: { title, tooltip, description, items }
        } = this.props;

        if (items.length === 0) {
            return null;
        }

        const itemsPrepared = items.map((item, i: number) => {
            return {
                nav: this.renderNavItem(item, i),
                content: this.renderNavContent(item, i)
            };
        });

        return (
            <Card theme="white" location="inner" stretched>
                <div className={b()}>
                    <div className={b('title')}>
                        <Title h3={title} />

                        {tooltip && (
                            <div className={b('hint-icon')}>
                                <HintIcon
                                    hint={tooltip}
                                    mixClassName={b('hint')}
                                    width={300}
                                />
                            </div>
                        )}
                    </div>

                    <div className={b('content')}>
                        <TabsPanes items={itemsPrepared} nav="top" />
                    </div>

                    {description && (
                        <div
                            className={b('description', ['text'])}
                            dangerouslySetInnerHTML={{ __html: description }}
                        />
                    )}
                </div>
            </Card>
        );
    }
}
