import React, { Component } from 'react';
import { throttle } from 'lodash';

import cn from 'client/utils/cn';
import LangContext from 'client/utils/lang-context';
import Logo from 'client/components/logo';
import MenuLinks from 'client/components/menu-links';
import MenuSite from 'client/components/menu-site';
import Nav from 'client/components/nav';
import 'client/components/visually-hidden/index.css';

import './index.css';

const b = cn('header');

interface Props {
    common: BunkerCommon;
    isMenuOpened: boolean;
    mainPage: boolean;
    onNavToggled: (opened: boolean) => void;
}

interface State {
    isPageScrolled: boolean;
}

class Header extends Component<Props, State> {
    constructor(props: Props) {
        super(props);

        this.state = {
            isPageScrolled: false
        };
    }

    componentDidMount(): void {
        window.addEventListener('scroll', throttle(this.onPageScroll, 300));
    }

    onPageScroll = (): void => {
        this.setState({ isPageScrolled: window.scrollY > 10 });
    };

    toggleMenu = (): void => {
        const { isMenuOpened, onNavToggled } = this.props;

        onNavToggled(!isMenuOpened);
    };

    static contextType = LangContext;

    render(): React.ReactNode {
        const {
            isMenuOpened,
            mainPage,
            common: { title, yandexMainPage, agenda, menu, report, links }
        } = this.props;
        const { dictionary } = this.context;

        const { isPageScrolled } = this.state;

        return (
            <div
                className={b({
                    theme: mainPage && !isPageScrolled && 'transparent',
                    'nav-opened': isMenuOpened
                })}
            >
                <header className={b('main')}>
                    <div className={b('main-content')}>
                        <Logo
                            theme={
                                mainPage && !isPageScrolled && !isMenuOpened
                                    ? 'white'
                                    : 'black'
                            }
                            title={title}
                            yandexMainPage={yandexMainPage}
                        />

                        <div className={b('controls')}>
                            <div className={b('links')}>
                                <MenuLinks links={links} view="header" />
                            </div>

                            <button
                                type="button"
                                className={b('nav', { opened: isMenuOpened })}
                                onClick={this.toggleMenu}
                                aria-expanded={isMenuOpened}
                            >
                                <span className="visually-hidden">
                                    {dictionary.menu}
                                </span>
                            </button>
                        </div>
                    </div>
                </header>

                <MenuSite menu={menu} report={report} agenda={agenda} />

                <Nav
                    menu={menu}
                    links={links}
                    report={report}
                    agenda={agenda}
                    isMenuOpened={isMenuOpened}
                />
            </div>
        );
    }
}

export default Header;
