import React, { Component } from 'react';
import { throttle } from 'lodash';

import cn from 'client/utils/cn';
import 'client/components/text/index.css';

import './index.css';

const b = cn('hint');

const enum HINT_DIRECTION_MOD {
    toLeft = 'to-left',
    toRight = 'to-right'
}

interface Props<T extends HTMLElement> {
    mixClassName: string;
    content?: string;
    parentNode: T | null;
    width: number;
}

interface State {
    direction: HINT_DIRECTION_MOD;
}

class Hint<T extends HTMLElement> extends Component<Props<T>, State> {
    constructor(props: Props<T>) {
        super(props);

        this.state = {
            direction: HINT_DIRECTION_MOD.toLeft
        };

        this.calcHintDirection = throttle(
            this.calcHintDirection.bind(this),
            300
        );
    }

    componentDidMount(): void {
        this.calcHintDirection();

        window.addEventListener('resize', this.calcHintDirection);
    }

    componentDidUpdate(prevProps: Props<T>): void {
        const { parentNode } = this.props;

        if (prevProps.parentNode !== parentNode) {
            this.calcHintDirection();
        }
    }

    componentWillUnmount(): void {
        window.removeEventListener('resize', this.calcHintDirection);
    }

    private tryToSetState(value: HINT_DIRECTION_MOD): void {
        const { direction } = this.state;

        if (direction !== value) {
            this.setState({
                direction: value
            });
        }
    }

    private calcHintDirection(): void {
        const { parentNode: ref, width } = this.props;
        const { direction } = this.state;
        const nodeRect = ref?.getBoundingClientRect();

        if (!nodeRect) {
            return;
        }

        const windowWidth = document.documentElement.clientWidth;

        // Пробуем открыть всегда сначала вправо
        const predictedRightEdge = nodeRect.left + width;

        if (predictedRightEdge < windowWidth) {
            this.tryToSetState(HINT_DIRECTION_MOD.toRight);
            return;
        }

        // Пробуем открыть влево, если не удалось открыть вправо
        const predictedLeftEdge = nodeRect.right - width;
        const MIN_LEFT_EDGE = 0;

        if (predictedLeftEdge > MIN_LEFT_EDGE) {
            this.tryToSetState(HINT_DIRECTION_MOD.toLeft);
            return;
        }

        // Если никуда нельзя открыть, открываем вправо
        if (direction !== HINT_DIRECTION_MOD.toRight) {
            this.tryToSetState(HINT_DIRECTION_MOD.toRight);
        }
    }

    render(): React.ReactNode {
        const { mixClassName, children, content } = this.props;

        if (!(children || content)) {
            return null;
        }

        const { direction } = this.state;

        return (
            <div className={b({ direction }, [mixClassName, 'text'])}>
                {content ? (
                    <span dangerouslySetInnerHTML={{ __html: content }} />
                ) : (
                    children
                )}
            </div>
        );
    }
}

export default Hint;
