import React, { Component } from 'react';
import Popup from 'reactjs-popup';

import 'client/components/popup/index.css';
import cn from 'client/utils/cn';
import 'client/components/text/index.css';
import 'client/components/visually-hidden/index.css';
import LangContext from 'client/utils/lang-context';
import Title from 'client/components/title';

import './index.css';

const b = cn('link');

interface Props {
    text?: string;
    title?: string;
    href?: string;
    target?: '_blank' | undefined;
    icon?: 'inside' | 'outside' | undefined;
    popup?: string;
    children?: React.ReactElement | React.ReactElement[];
    bg?: boolean;
}

interface State {
    isPopupOpen: boolean;
}

export default class Link extends Component<Props, State> {
    constructor(props: Props) {
        super(props);

        this.state = {
            isPopupOpen: false
        };
    }

    hasScrollbar = (): boolean => {
        return Boolean(
            window.innerWidth -
                document.getElementsByTagName('html')[0].clientWidth
        );
    };

    openPopup = (): void => {
        if (this.hasScrollbar()) {
            document.body.classList.add('scrollbar__yes');
        }
        document.body.classList.add('popup__opened');
        this.setState({ isPopupOpen: true });
    };

    closePopup = (): void => {
        document.body.classList.remove('popup__opened');
        document.body.classList.remove('scrollbar__yes');
        this.setState({ isPopupOpen: false });
    };

    static contextType = LangContext;

    renderLinkContent = (): React.ReactNode => {
        const { text, icon, children } = this.props;

        return (
            <>
                {text || children}

                {icon && <span className={b('icon', { type: icon })} />}
            </>
        );
    };

    render(): React.ReactNode {
        const { href, target, popup, bg, title } = this.props;
        const { isPopupOpen } = this.state;
        const { dictionary } = this.context;

        if (!href && !popup) {
            return null;
        }

        if (popup) {
            return (
                <>
                    <button
                        className={b()}
                        type="button"
                        onClick={this.openPopup}
                    >
                        {this.renderLinkContent()}
                    </button>

                    <Popup
                        open={isPopupOpen}
                        closeOnDocumentClick
                        onClose={this.closePopup}
                        className={isPopupOpen ? 'opened-popup' : ''}
                    >
                        <div className="popup-modal">
                            <div className="popup-content__inner">
                                <Title h2={title} />

                                <div
                                    className="text"
                                    dangerouslySetInnerHTML={{ __html: popup }}
                                />

                                <button
                                    className="popup-modal__close"
                                    type="button"
                                    onClick={this.closePopup}
                                >
                                    <div className="visually-hidden">
                                        {dictionary.closebutton}
                                    </div>
                                </button>
                            </div>
                        </div>
                    </Popup>
                </>
            );
        }

        return (
            <a href={href} target={target} className={b({ bg })}>
                {this.renderLinkContent()}
            </a>
        );
    }
}
