import React, { Component } from 'react';

import cn from 'client/utils/cn';
import { MenuItem, MenuListItem } from 'client/bundles/types';
import LangContext from 'client/utils/lang-context';

import './index.css';

const b = cn('menu-site-sections');

interface Props {
    items: MenuListItem[];
    agenda?: BunkerCommon['agenda'];
    view: MenuListItem['view'];
}

interface State {
    indexSectionOpened?: number;
}

class MenuSiteSections extends Component<Props, State> {
    private root = React.createRef<HTMLDivElement>();

    constructor(props: Props) {
        super(props);

        this.state = {};
    }

    componentDidMount(): void {
        window.addEventListener('mouseup', this.checkIfClickedOutside);
    }

    componentWillUnmount(): void {
        window.removeEventListener('mouseup', this.checkIfClickedOutside);
    }

    checkIfClickedOutside = (e: Event): void => {
        const { indexSectionOpened } = this.state;

        // Меню отрыто и клик пришелся не в меню
        if (
            indexSectionOpened !== undefined &&
            this.root.current &&
            e.target instanceof Node &&
            !this.root.current.contains(e.target)
        ) {
            this.setState({ indexSectionOpened: undefined });
            // Чтобы не сработал какой-нибудь клик, если пользователь попал на ссылку в документе
            e.preventDefault();
        }
    };

    toggleSection = (index: number): void => {
        const { view } = this.props;
        const { indexSectionOpened } = this.state;

        if (view === 'header') {
            this.setState({
                indexSectionOpened:
                    indexSectionOpened === index ? undefined : index
            });
        }
    };

    static contextType = LangContext;

    renderItems = (item: MenuItem, index: number): React.ReactNode => {
        return (
            <li key={index} className={b('section-item')}>
                <a
                    className={b('section-link', { active: item.isCurrent })}
                    href={item.url}
                    dangerouslySetInnerHTML={{ __html: item.title }}
                />
            </li>
        );
    };

    renderTitleContent = (item: MenuListItem): React.ReactNode => {
        return (
            <>
                <span className={b('title-text')}>{item.subtitle}</span>
                <span className={b('title-control')} />
            </>
        );
    };

    renderAgenda = (): React.ReactNode => {
        const { agenda } = this.props;

        if (!agenda) {
            return null;
        }

        return (
            <li className={b('list-item', { agenda: true })}>
                <span className={b('title')}>
                    <a
                        className={b('title-link', {
                            active: agenda.isCurrent
                        })}
                        href={agenda.url}
                    >
                        <span className={b('title-text')}>{agenda.title}</span>
                    </a>
                </span>
            </li>
        );
    };

    renderTitle = (item: MenuListItem, index: number): React.ReactNode => {
        const { view } = this.props;
        const content = this.renderTitleContent(item);
        const { indexSectionOpened } = this.state;

        return view === 'header' ? (
            <button
                type="button"
                className={b('title')}
                onClick={(): void => this.toggleSection(index)}
                aria-haspopup="true"
                aria-expanded={indexSectionOpened === index}
            >
                {content}
            </button>
        ) : (
            <span className={b('title')}>{content}</span>
        );
    };

    renderSection = (item: MenuListItem, index: number): React.ReactNode => {
        const { indexSectionOpened } = this.state;

        return (
            <li
                key={index}
                className={b('list-item', {
                    opened: indexSectionOpened === index,
                    active: item.isCurrent
                })}
            >
                {this.renderTitle(item, index)}

                <ul className={b('section')}>
                    {item.items.map(this.renderItems)}
                </ul>
            </li>
        );
    };

    render(): React.ReactNode {
        const { items, view } = this.props;

        return (
            <nav className={b({ view })} ref={this.root}>
                <ul className={b('list')}>
                    {this.renderAgenda()}

                    {items.map(this.renderSection)}
                </ul>
            </nav>
        );
    }
}

export default MenuSiteSections;
