import React, { Component } from 'react';

import cn from 'client/utils/cn';

import './index.css';

const b = cn('page-menu');

interface Props {
    menu: MenuItem[];
    gap: number;
}

interface State {
    selectedItem: string;
}

// Пользователь увидел минимум 100px блока у верхней границы экрана
const VISIBLE_TOP_HEIGHT = 100;

// Пользователь увидел минимум 55% блока у нижней границы экрана
const VISIBLE_BOTTOM_HEIGHT = '55%';

const STICKY_HEADER_HEIGHT = 126;
const VIEWPORT_MARGIN_TOP = `-${STICKY_HEADER_HEIGHT + VISIBLE_TOP_HEIGHT}px`;
const VIEWPORT_MARGIN_BOTTOM = `-${VISIBLE_BOTTOM_HEIGHT}`;

export default class PageMenu extends Component<Props, State> {
    private observer: IntersectionObserver;

    constructor(props: Props) {
        super(props);

        this.state = {
            selectedItem: ''
        };
    }

    componentDidMount(): void {
        if (window.location.href.indexOf('#')) {
            const id = window.location.hash.split('#')[1];

            this.setState({ selectedItem: id });
        }

        this.observer = new IntersectionObserver(this.observeMenuItems, {
            rootMargin: `${VIEWPORT_MARGIN_TOP} 0px ${VIEWPORT_MARGIN_BOTTOM} 0px`
        });

        this.initObserveMenu();
    }

    componentWillUnmount(): void {
        this.observer.disconnect();
    }

    initObserveMenu = (): void => {
        const { menu } = this.props;
        menu.forEach((item) => {
            const element = document.getElementById(`${item.id}-content`);
            if (element) {
                this.observer.observe(element);
            }
        });
    };

    observeMenuItems = (entries: IntersectionObserverEntry[]): void => {
        entries.forEach((entry) => {
            if (entry.isIntersecting) {
                const selectedItem = entry.target.id.replace('-content', '');
                this.setState({ selectedItem });
            }
        });
    };

    onItemClick = (id: string): void => {
        if (typeof window === 'undefined') {
            return;
        }

        const block = window.document.getElementById(id);

        if (!block) {
            return;
        }

        block.scrollIntoView({ block: 'start', behavior: 'smooth' });

        const { selectedItem } = this.state;

        if (id === selectedItem) {
            return;
        }

        this.setState({ selectedItem: id });

        const url = new URL(window.location.href);
        const currentUrl = url.href.split('#')[0];

        if (window.history && window.history.replaceState) {
            window.history.replaceState(null, '', `#${id}`);
        } else {
            window.location.href = `${currentUrl}#${id}`;
        }
    };

    renderItem = (item: MenuItem, i: number): React.ReactNode => {
        const { selectedItem } = this.state;

        if (!item.titlenav || !item.id) {
            return null;
        }

        return (
            <li className={b('list-item')} key={`menu-item-${i}`}>
                <button
                    type="button"
                    className={b('button', {
                        active: selectedItem === item.id
                    })}
                    onClick={(): void => this.onItemClick(item.id)}
                >
                    {item.titlenav}
                </button>
            </li>
        );
    };

    render(): React.ReactNode {
        const { menu, gap } = this.props;

        if (!menu || menu.length === 0) {
            return null;
        }

        return (
            <nav
                className={b()}
                style={{ marginLeft: `${gap}px` }}
                aria-hidden="true"
            >
                <ul className={b('list')}>{menu.map(this.renderItem)}</ul>
            </nav>
        );
    }
}
