import React, { Component } from 'react';
import { throttle } from 'lodash';

import cn from 'client/utils/cn';
import PageMenu from 'client/components/page-menu';
import Title from 'client/components/title';
import 'client/components/text/index.css';
import 'client/components/wiki-text/index.css';

import './index.css';

const b = cn('page');

interface PageProps {
    children: React.ReactNode | React.ReactNode[];
    menu?: MenuItem[];
}

interface PageSectionProps {
    children?: React.ReactNode;
    isMainPage?: boolean | undefined;
    theme: 'white' | 'light-grey' | 'grey' | 'black';
    offsetTop?: boolean;
    offsetBottom?: boolean;
    breadcrumb?: string;
    preTitle?: string;
    h1?: string;
    h2?: string;
    url?: string;
    icon?: 'helping-hand' | 'practicum' | undefined;
    text?: string;
    id?: string;
}

interface State {
    menuGap?: number;
}

interface StateSection {
    isItemOpened: boolean;
}

export class Page extends Component<PageProps, State> {
    private ref = React.createRef<HTMLDivElement>();

    constructor(props: PageProps) {
        super(props);

        this.state = {
            menuGap: undefined
        };
        this.setMenuGap = throttle(this.setMenuGap.bind(this), 100);
    }

    componentDidMount(): void {
        this.setMenuGap();

        window.addEventListener('resize', this.setMenuGap);
    }

    private setMenuGap(): void {
        const nodeRect = this.ref?.current?.getBoundingClientRect();

        if (nodeRect) {
            this.setState({ menuGap: Math.floor(nodeRect.x) });
        }
    }

    render(): React.ReactNode {
        const { children, menu } = this.props;
        const { menuGap } = this.state;

        return (
            <main className={b()}>
                <div
                    className={b('menuanchor')}
                    ref={this.ref}
                    aria-hidden="true"
                />

                {menuGap && menu ? (
                    <PageMenu menu={menu} gap={menuGap} />
                ) : null}

                {children}
            </main>
        );
    }
}

export class PageSection extends Component<PageSectionProps, StateSection> {
    constructor(props: PageSectionProps) {
        super(props);

        this.state = {
            isItemOpened: false
        };
    }

    toggleItem = (): void => {
        const { isItemOpened } = this.state;

        this.setState({ isItemOpened: !isItemOpened });
    };

    renderTitle = (): React.ReactNode => {
        const { h2, url, icon } = this.props;

        return (
            <div className={b('title')}>
                <Title h2={h2} url={url} icon={icon} />
            </div>
        );
    };

    renderContent = (): React.ReactNode => {
        const { text, children } = this.props;

        return (
            <>
                {text && (
                    <div
                        className={b('text', ['text'])}
                        dangerouslySetInnerHTML={{
                            __html: text
                        }}
                    />
                )}

                {children}
            </>
        );
    };

    render(): React.ReactNode {
        const {
            isMainPage,
            theme,
            offsetTop,
            offsetBottom,
            breadcrumb,
            h1,
            h2,
            id
        } = this.props;
        const { isItemOpened } = this.state;
        const mods = {
            theme,
            'offset-top': offsetTop,
            'offset-bottom': offsetBottom,
            opened: isItemOpened ? 'yes' : 'no',
            'not-expand': !h2,
            main: isMainPage
        };

        return (
            <section className={b('section', mods)}>
                {id && <div className={b('anchor')} id={id} />}

                <div className={b('section-inner')}>
                    <div
                        className={b('section-content')}
                        id={id && `${id}-content`}
                    >
                        {breadcrumb && (
                            <div
                                className={b('breadcrumb')}
                                dangerouslySetInnerHTML={{ __html: breadcrumb }}
                            />
                        )}

                        {h1 && (
                            <>
                                <div className={b('title')}>
                                    <Title h1={h1} />
                                </div>

                                {this.renderContent()}
                            </>
                        )}

                        {h2 && (
                            <>
                                {this.renderTitle()}

                                <button
                                    className={b('button-title')}
                                    type="button"
                                    onClick={this.toggleItem}
                                    aria-expanded={isItemOpened}
                                >
                                    {this.renderTitle()}

                                    <span className={b('icon-expand')} />
                                </button>

                                <div className={b('section-content-inner')}>
                                    {this.renderContent()}
                                </div>
                            </>
                        )}

                        {!h1 && !h2 && this.renderContent()}
                    </div>
                </div>
            </section>
        );
    }
}
