import React, { Component } from 'react';

import cn from 'client/utils/cn';
import Card from 'client/components/card';
import CardFact from 'client/components/card-fact';
import ImageCard from 'client/components/image-card';
import Link from 'client/components/link';
import Title from 'client/components/title';
import TabsPanes from 'client/components/tabs-panes';
import SectionGridRow from 'client/components/section-grid-row';

import './index.css';

const b = cn('section-card-tabs');

interface Props {
    items: CardTabsItem[];
    theme: 'white' | 'light-grey';
    cardLinkText?: string;
}

export default class SectionCardTabs extends Component<Props> {
    renderImage = (item: CardTabsItem): React.ReactNode => {
        if (!item.image) {
            return null;
        }

        return (
            <div className={b('image')}>
                <ImageCard src={item.image} title={item.title} hidden />
            </div>
        );
    };

    renderBio = (item: CardTabsItem): React.ReactNode => {
        if (!item.bio) {
            return null;
        }

        return (
            <div className={b('segalovich')}>
                <div
                    className={b('text')}
                    dangerouslySetInnerHTML={{ __html: item.bio }}
                />

                <div className={b('photo')} />
            </div>
        );
    };

    renderFacts = (item: CardTabsItem, index: number): React.ReactNode => {
        if (!item.facts || item.facts.length === 0) {
            return null;
        }

        return (
            <div className={b('facts')}>
                <SectionGridRow>
                    {item.facts.map((fact) => (
                        <CardFact
                            theme="light-grey"
                            key={index}
                            location="inner"
                            title={fact.title}
                            text={fact.text}
                            stretched
                        />
                    ))}
                </SectionGridRow>
            </div>
        );
    };

    renderLink = (item: CardTabsItem): React.ReactNode => {
        const { cardLinkText } = this.props;

        if (!item.url || !cardLinkText) {
            return null;
        }

        return (
            <div className={b('link')}>
                <Link
                    text={cardLinkText}
                    href={item.url}
                    target="_blank"
                    icon="outside"
                />
            </div>
        );
    };

    renderNavItem = (item: CardTabsItem): React.ReactNode => {
        return (
            <div className={b('nav')}>
                {item.period && (
                    <span className={b('nav-period')}>{item.period}</span>
                )}
                <span className={b('nav-title')}>{item.title}</span>
            </div>
        );
    };

    renderNavContent = (item: CardTabsItem, index: number): React.ReactNode => {
        return (
            <div className={b('content')}>
                <div className={b('title')}>
                    <Title h3={item.title} url={item.titleUrl} />
                </div>

                <div
                    className={b('text')}
                    dangerouslySetInnerHTML={{ __html: item.text }}
                />

                {this.renderFacts(item, index)}

                {this.renderBio(item)}

                {this.renderImage(item)}

                {this.renderLink(item)}
            </div>
        );
    };

    render(): React.ReactNode {
        const { items, theme } = this.props;

        if (items.length === 0) {
            return null;
        }

        const itemsPepared = items.map((item, index) => {
            return {
                nav: this.renderNavItem(item),
                content: this.renderNavContent(item, index)
            };
        });

        return (
            <Card theme={theme} wide>
                <div className={b({ theme })}>
                    <TabsPanes items={itemsPepared} nav="left" theme={theme} />
                </div>
            </Card>
        );
    }
}
