import React, { Component } from 'react';

import cn from 'client/utils/cn';

import './index.css';

const b = cn('tabs-panes');

type Item = {
    nav: React.ReactNode;
    content: React.ReactNode;
};

interface Props {
    isActive?: number;
    items: Item[];
    nav: 'bottom' | 'top' | 'left';
    theme?: 'white' | 'light-grey' | undefined;
}

interface State {
    clickedItem: number;
}

export default class TabsPanes extends Component<Props, State> {
    private tabRefs: React.RefObject<HTMLButtonElement>[] = [];

    constructor(props: Props) {
        super(props);

        const { isActive } = this.props;

        this.state = {
            clickedItem: isActive || 0
        };
    }

    componentDidMount(): void {
        window.addEventListener('keyup', this.onKeyUp);
    }

    onKeyUp = (e: KeyboardEvent): void => {
        const { items } = this.props;
        const { clickedItem } = this.state;
        if (document.activeElement !== this.tabRefs[clickedItem].current) {
            return;
        }

        let newClickedItem = clickedItem;
        if (e.key === 'ArrowRight') {
            const nextIndex = clickedItem + 1;
            const lastIndex = items.length - 1;
            newClickedItem = nextIndex > lastIndex ? lastIndex : nextIndex;
        } else if (e.key === 'ArrowLeft') {
            const prevIndex = clickedItem - 1;
            newClickedItem = prevIndex < 0 ? 0 : prevIndex;
        }

        this.setState({ clickedItem: newClickedItem });
        this.tabRefs[newClickedItem].current?.focus();
    };

    onItemClicked = (index: number): void => {
        const { clickedItem } = this.state;

        if (index === clickedItem) {
            return;
        }

        this.setState({ clickedItem: index });
    };

    renderContentItem = (item: Item, index: number): React.ReactNode => {
        const { clickedItem } = this.state;
        const isVisible = clickedItem === index;

        return (
            <div
                key={index}
                role="tabpanel"
                id={`tabpanel-${index}`}
                tabIndex={0}
                className={b('content-item', { visible: isVisible })}
            >
                {item.content}
            </div>
        );
    };

    renderNavItem = (item: Item, index: number): React.ReactNode => {
        this.tabRefs[index] = this.tabRefs[index] || React.createRef();

        const { clickedItem } = this.state;
        const isActive = clickedItem === index;

        return (
            <button
                key={index}
                ref={this.tabRefs[index]}
                className={b('button', { active: isActive })}
                type="button"
                role="tab"
                id={`tab-${index}`}
                tabIndex={isActive ? undefined : -1}
                aria-controls={`tabpanel-${index}`}
                onClick={(): void => this.onItemClicked(index)}
                aria-selected={isActive ? 'true' : 'false'}
            >
                <div className={b('button-text')}>{item.nav}</div>
            </button>
        );
    };

    render(): React.ReactNode {
        const { items, nav, theme } = this.props;

        if (items.length === 0) {
            return null;
        }

        return (
            <div className={b({ position: nav, theme })}>
                {items.length > 1 && (
                    <div className={b('nav')}>
                        <div className={b('nav-items')} role="tablist">
                            {items.map(this.renderNavItem)}
                        </div>
                    </div>
                )}

                <div className={b('content')}>
                    {items.map(this.renderContentItem)}
                </div>
            </div>
        );
    }
}
