import { NextFunction, Request, Response } from 'express';

import { getBunkerPageData } from 'server/utils/bunker-utils';
import {
    getCommonPageData,
    getMenu,
    getPreparedSections
} from 'server/utils/page-utils';
import getFormattedText, { formatTextsToWiki } from 'server/utils/wiki-utils';

export default async function employeesController(
    req: Request,
    res: Response,
    next: NextFunction
): Promise<void> {
    const pageSection = 'investing-in-people';
    const pageName = 'employees';
    const { bunker, lang } = req;
    const pageData = getBunkerPageData(lang, bunker, pageName);
    const { page } = pageData;

    if (!page.sections) {
        next();

        return;
    }

    const sectionsPrepared = getPreparedSections(page.sections);

    page.sections = sectionsPrepared;
    page.menu = getMenu(sectionsPrepared);

    const diversity = page.sections
        .diversity as unknown as BunkerPageEmployees['sections']['diversity'];
    const { left: diversityLeft, right: diversityRight } = diversity.items;
    if (diversityLeft.length > 0) {
        diversity.items.left = await formatTextsToWiki(diversityLeft);
    }
    if (diversityRight.length > 0) {
        diversity.items.right = await formatTextsToWiki(diversityRight);
    }

    const developing = page.sections
        .developing as unknown as BunkerPageEmployees['sections']['developing'];
    const { left: developingLeft, right: developingRight } = developing.items;
    if (developingLeft.popup) {
        developing.items.left.popup = await getFormattedText(
            developingLeft.popup
        );
    }
    if (developingRight.popup) {
        developing.items.right.popup = await getFormattedText(
            developingRight.popup
        );
    }

    const wellbeing = page.sections
        .wellbeing as unknown as BunkerPageEmployees['sections']['wellbeing'];

    const { finances, comfort } = wellbeing;
    if (finances.items.length > 0) {
        wellbeing.finances.items = await formatTextsToWiki(finances.items);
    }
    if (comfort.items.length > 0) {
        wellbeing.comfort.items = await formatTextsToWiki(comfort.items);
    }

    res.renderBundle('common', pageName, {
        ...getCommonPageData(req, res, pageName, pageSection),
        ...pageData
    });
}
