import { ErrorRequestHandler, RequestHandler, Router } from 'express';

import agendaController from 'server/controllers/agenda';
import faqController from 'server/controllers/faq';
import downloadController from 'server/controllers/download-center/download';
import webcastController from 'server/controllers/download-center/webcast';
import documentController from 'server/controllers/download-center/document';
import mainController from 'server/controllers/main';
import notFoundController from 'server/controllers/not-found';
import convenienceController from 'server/controllers/quality-of-life/convenience';
import accessibleController from 'server/controllers/quality-of-life/accessible';
import educationController from 'server/controllers/investing-in-people/education';
import partnersController from 'server/controllers/investing-in-people/partners';
import employeesController from 'server/controllers/investing-in-people/employees';
import anticorruptionController from 'server/controllers/ethics-integrity/anticorruption';
import qualityController from 'server/controllers/ethics-integrity/quality';
import procurementController from 'server/controllers/ethics-integrity/procurement';
import securityController from 'server/controllers/ethics-integrity/security';
import carbonController from 'server/controllers/environmental-impact/carbon';
import energyController from 'server/controllers/environmental-impact/energy';
import wasteController from 'server/controllers/environmental-impact/waste';
import {
    bunker,
    cookieParser,
    helmet,
    i18n,
    httpUatraits,
    tld,
    tvm,
    xForwardedForFix,
    yandexCsp,
    yandexuid
} from 'server/middlewares';

const middlewares: Array<RequestHandler | ErrorRequestHandler> = [
    bunker,
    cookieParser,
    tld,
    i18n,
    tvm,
    helmet,
    yandexCsp, // Depends on cookieParser
    xForwardedForFix,
    httpUatraits,
    yandexuid // Depends on cookieParser and httpUatraits
];

export default function (router: Router): void {
    router.get('/', middlewares, mainController);
    router.get('/agenda', middlewares, agendaController);
    router.get('/faq', middlewares, faqController);
    router.get('/download-center', middlewares, downloadController);
    router.get('/download-center/webcast', middlewares, webcastController);
    router.get('/download-center/:docPath', middlewares, documentController);
    router.get(
        '/quality-of-life/convenience-quality-safety-of-services',
        middlewares,
        convenienceController
    );
    router.get(
        '/quality-of-life/accessible-environment',
        middlewares,
        accessibleController
    );
    router.get(
        '/investing-in-people/education-for-all',
        middlewares,
        educationController
    );
    router.get(
        '/investing-in-people/yandex-service-partners',
        middlewares,
        partnersController
    );
    router.get(
        '/investing-in-people/yandex-employees',
        middlewares,
        employeesController
    );
    router.get(
        '/ethics-integrity/responsible-procurement',
        middlewares,
        procurementController
    );
    router.get(
        '/ethics-integrity/information-security-data-privacy',
        middlewares,
        securityController
    );
    router.get(
        '/ethics-integrity/anti-corruption-and-antitrust',
        middlewares,
        anticorruptionController
    );
    router.get(
        '/ethics-integrity/quality-content',
        middlewares,
        qualityController
    );
    router.get(
        '/environmental-impact/energy-efficiency',
        middlewares,
        energyController
    );
    router.get(
        '/environmental-impact/carbon-footprint',
        middlewares,
        carbonController
    );
    router.get(
        '/environmental-impact/packaging-waste',
        middlewares,
        wasteController
    );

    router.use(middlewares, notFoundController);
}
