export type PageName = 'main' | 'structure' | 'faq' | 'emailAlerts' | 'analysts' | 'financials' | 'contacts' |
    'events' | 'meetings' | 'sec' | 'stocks' | 'directors' | 'notFound' | 'pressReleases' | 'sustainability' |
    'ourPrinciples' | 'keyDocuments' | 'overview' | 'management' | 'faqGovernance' | 'contactsGovernance' |
    'keyBusinesses' | 'webcast' | 'latestPresentation' | 'bonds';

export type mainDocsName = 'databook' | 'latest-annual-report' | 'latest-presentation';

export interface MenuItem {
    id: string;
    url: string;
    title: string;
    isCurrent?: boolean;
}

export type PhoneNumber = string;
export type EmailData = string;
export type Address = string;

export interface OfficeContactsItem {
    title?: string;
    address?: Address;
    phone?: PhoneNumber;
    fax?: PhoneNumber;
    email?: EmailData;
}

export interface OfficeContacts {
    ir: OfficeContactsItem;
    pr: OfficeContactsItem;
    adv: OfficeContactsItem;
    sustainability: OfficeContactsItem;
    secretary: OfficeContactsItem;
    headquaters: OfficeContactsItem;
}
export interface ContactSectionAddress {
    name?: string;
    text?: string;
    address?: string;
    contact: OfficeContactsItem;
    contacts: OfficeContacts;
}

export interface ContactSectionForm {
    title?: string;
    text?: string;
    buttonText?: string;
    buttonLink?: string;
}

export interface PrefooterData {
    amsterdamContacts: ContactSectionAddress;
    moscowContacts: ContactSectionAddress;
    ask: ContactSectionForm;
    subscribe: ContactSectionForm;
    additionalInfo: string;
}

export interface MetrikaData {
    metrikaId?: number;
    metrikaOptions: {
        clickmap?: boolean,
        trackLinks?: boolean,
        webvisor?: boolean,
        accurateTrackBounce?: boolean
    };
}

export interface BundleProps {
    data: BundleData;
}

interface ServiceLink {
    href: string;
    title: string;
}

export interface Service {
    company?: ServiceLink;
    vacancies?: ServiceLink;
    tech?: ServiceLink;
    adv?: ServiceLink;
    ir?: ServiceLink;
    events?: ServiceLink;
    science?: ServiceLink;
    isActive?: boolean;
}

export type ServiceItem = 'company' | 'vacancies' | 'tech' | 'adv' | 'ir' | 'events' | 'science';

export interface Analyst {
    firm: string;
    person: string;
    email: string;
}

export interface BunkerAnalysts {
    title: string;
    analysts: Analyst[];
    text: string;
}

export const COMMITTEE_TYPES = ['all', 'governance', 'interest', 'audit', 'investment', 'nominating', 'compensation'];

export type DirectorCommittee = typeof COMMITTEE_TYPES[number];

export interface DirectorData {
    person: string;
    role: string;
    photo: string;
    bio: string;
    chairpersonCommittees: DirectorCommittee[];
    memberCommittees: DirectorCommittee[];
}

export const COMMITTEES: Record<DirectorCommittee, string> = {
    all: 'All',
    governance: 'Corporate Governance',
    interest: 'Public Interest',
    audit: 'Audit',
    investment: 'Investment',
    nominating: 'Nominating',
    compensation: 'Compensation'
};

export interface BunkerDirectors {
    title: string;
    isWiki: boolean;
    directors: DirectorData[];
}

export interface BunkerDocuments {
    title: string;
    text: string;
    corporateDocs: Doc[];
    chartersDocs: Doc[];
}

export interface BunkerDocLink {
    link: string;
}

export interface BunkerWebcast {
    title?: string;
    link?: string;
}

export const SUSTAINABILITY_TYPES = ['report-en', 'report-ru', 'presentation', 'webcast'] as const;

export type BunkerSustainabilityLinkType = typeof SUSTAINABILITY_TYPES[number];

export interface BunkerSustainability {
    title?: string;
    text?: string;
    docs?: Doc[];
    'report-en'?: BunkerDocLink;
    'report-ru'?: BunkerDocLink;
    'presentation'?: BunkerDocLink;
    'webcast'?: BunkerWebcast;
}

export interface BunkerBonds {
    title: string;
    disclaimers: BunkerBondsDisclaimer[];
    docs: Doc[];
}

export interface BunkerBondsDisclaimer {
    text: string;
    html: string;
    buttons: {
        accept: {
            text: string;
        };
        decline: {
            text: string;
        }
    };
}

export interface BunkerLatestPresentation {
    title?: string;
    text?: string;
    docs?: Doc[];
}

export interface BunkerOverview {
    title: string;
    isWiki: boolean;
    text: string;
}

export interface BunkerContacts {
    title: string;
}

export interface PersonInfo {
    photo: string;
    name: string;
    position: string;
    bio: string;
}

export interface BunkerManagement {
    title: string;
    isWiki: boolean;
    list: PersonInfo[];
}

export interface BunkerGovernance {
    title: string;
    documents?: BunkerDocuments;
    overview?: BunkerOverview;
    contacts?: BunkerContacts;
    directors?: BunkerDirectors;
    faq?: FaqListItem;
    management?: BunkerManagement;
}

export interface TypeMeetingData {
    title: string;
    list?: OccasionData[];
}

export type TypeMeetingId = 'all' | 'annual' | 'extraordinary';

export interface OccasionData {
    title: string;
    location?: string;
    docs: Doc[];
    dateStart: string;
    dateEnd?: string;
    timeStart?: Time;
    timeEnd?: Time;
}

export interface BunkerFinancialQuarter {
    title: string;
    date: string;
    docs: Doc[];
    url: string;
    html?: string;
    linkPdf?: string;
}

export const REPORT_NAMES = ['q1', 'q2', 'q3', 'q4', 'reaffirms'];

export type BunkerFinancialType = typeof REPORT_NAMES[number];

export type BunkerFinancialYear = {
    [key in BunkerFinancialType]: BunkerFinancialQuarter;
};

export interface BunkerMeetings {
    title: string;
    pageName: PageName;
    egm: TypeMeetingData;
    agm: TypeMeetingData;
}

export type DocType = 'presentation' | 'slides' | 'webcast';

export interface Doc {
    title: string;
    type?: DocType;
    link?: string;
    format?: string;
    description?: string;
}

export interface SecFiling {
    title: string;
    formType: string;
    pubDate: string;
    periodDate?: string;
    filingGroup: string;
    company: string;
    issuer: string;
    guid: string;
    documentFormatFiles: Doc[];
    xbrlSequence?: Doc[];
}

export interface Time {
    value?: string;
    zone?: string;
}

export interface EventsYear {
    title: string;
    currentYearData: OccasionData[];
}

export interface FaqQuestionItem {
    question?: string;
    answer: string;
}

export interface FaqListItem {
    title: string;
    isWiki: boolean;
    questions: FaqQuestionItem[];
}

export interface BunkerFaq {
    title: string;
    isWiki: boolean;
    list: FaqListItem[];
}

export interface BunkerPrinciplesData {
    title: string;
    imageUrl: string;
    isWiki: boolean;
    list: string[];
}

export interface BunkerPrinciples {
    [key: string]: BunkerPrinciplesData;
}

export const enum BunkerKeyBusinessesSubnodeType {
    Nope = 'nope',
    Segment = 'segment',
    Service = 'service'
}

export interface BunkerKeyBusinessesNode {
    title: string;
    content: string;
    'subnodes-type': BunkerKeyBusinessesSubnodeType;
    subnodes: BunkerKeyBusinessesNode[];
}

interface BunkerKeyBusinesses {
    title: string;
    isWiki: boolean;
    note: string;
    structure: BunkerKeyBusinessesNode[];
}

interface BunkerContactsGovernance {
    title: string;
}

export interface SharesIncludingData {
    classA: string;
    classB: string;
}

export interface SharesData {
    total: string;
    including?: SharesIncludingData;
    excluding: string;
}

export interface ShareholderPercents {
    title: string;
    eo: string;
    vp: string;
}
export interface ShareholdersData {
    preIPO: ShareholderPercents;
    volozh: ShareholderPercents;
    otherDirectors: ShareholderPercents;
    otherPreIPO: ShareholderPercents;
    publicInvestors: ShareholderPercents;
    total: ShareholderPercents;
}

export interface BunkerStructure {
    title?: string;
    date: string;
    shares: SharesData;
    shareholders: ShareholdersData;
}

export interface MainLink {
    title: string;
    url: string;
    isTargetBlank?: boolean;
}

interface MapNote {
    office: string;
    location: string;
}

interface MainGeographyData {
    isVisible: boolean;
    countries: Array<{
        country: string;
        type: 'office' | 'location';
        description: string;
        x: number;
        y: number;
    }>;
    'office-badge': string;
    'location-badge': string;
    note: MapNote;
}

export interface MainInfoData {
    isVisible: boolean;
    title: string;
    text: string;
    link: string;
}

export interface ProfileData {
    title: string;
    text: string;
}

export interface BunkerMain {
    title: string;
    links: MainLink[];
    history: MainInfoData;
    geography: MainGeographyData;
    principles: MainInfoData;
}

export interface FinancialsYear {
    title: string;
    reports: BunkerFinancialYear;
}

export interface BunkerEvents {
    [key: string]: OccasionData[];
}

export interface BunkerFinancials {
    [key: string]: BunkerFinancialYear;
}

export interface SecFilingType {
    groupID: string;
    groupName: string;
    groupFilings: string[];
}

export interface PostTag {
    slug: string;
    url: string;
}

export interface PostImage {
    orig?: { fullPath: string };
}

export interface PressReleaseItemData {
    url: string;
    title: string;
    date: string;

    markup?: string;
    annotation?: string;
    socialImage?: string;
    tagsArray: PostTag[];
    description: string;
    keywords: string;

    ignoreInIndex: boolean;
    ignoreByWidget: boolean;
    financialResults: boolean;
    isMigrated?: boolean;
}

export interface BunkerData {
    common: {
        menu: {
            items: MenuItem[]
        };
        prefooter: PrefooterData;
    };
    libs: {
        'sec_groups': {
            items: SecFilingType[];
        }
    };
    pages: {
        analysts?: BunkerAnalysts;
        directors?: BunkerDirectors;
        documents?: BunkerDocuments;
        governance?: BunkerGovernance;
        meetings?: BunkerMeetings;
        events?: BunkerEvents;
        financials?: BunkerFinancials;
        faq?: BunkerFaq;
        structure?: BunkerStructure;
        principles?: BunkerPrinciples;
        main?: BunkerMain;
        'key-businesses'?: BunkerKeyBusinesses;
        'latest-presentation'?: BunkerDocLink;
        'latest-annual-report'?: BunkerDocLink;
        databook?: BunkerDocLink;
        sustainability?: BunkerSustainability;
        bonds?: BunkerBonds;
    };
}

export interface FilingType {
    id: string;
    value: string;
}

export interface PageSecProps {
    groupId: string;
    currentYear: string;
    years: string[];
    types: FilingType[];
    yearFilings?: SecFiling[];
    singleFiling?: SecFiling;
    sectionTitle?: string;
}

export interface PageStructureProps {
    title: string;
    structure?: BunkerStructure;
    pageName: PageName;
}

export interface PageEventsProps {
    title: string;
    years: string[];
    currentYear: string;

    yearEvents?: OccasionData[];
}

export interface PageFinancialsProps {
    title: string;
    years: string[];
    currentYear: string;

    yearReports?: BunkerFinancialYear;
    quarterReport?: BunkerFinancialQuarter;
}

export interface PagePressReleasesProps {
    title?: string;
    items?: PressReleaseItemData[];
    item?: PressReleaseItemData;
    currentYear: string;
    years?: string[];
}

export interface PagePrinciplesProps {
    title: string;
    section: BunkerPrinciplesData;
    sectionId: string;
}

export interface PageMainProps {
    title: string;
    stocksTeaserUrl: string;
    profile: ProfileData;
    principles: MainInfoData;
    geography: MainGeographyData;
    history: MainInfoData;
    links: MainLink[];
    pressReleases: PressReleaseItemData[];
    lastFinancialReports: BunkerFinancialYear;
}

export interface PageDirectorsProps {
    title: string;
    list: DirectorData[];
    selectedCommittee?: DirectorCommittee;
}

export interface BundleData {
    readonly analysts: BunkerAnalysts;
    readonly documents: BunkerDocuments;
    readonly overview: BunkerOverview;
    readonly governance: BunkerGovernance;
    readonly management: BunkerManagement;
    readonly meetings: BunkerMeetings;
    readonly faq: BunkerFaq;
    readonly faqGovernance: FaqListItem;
    readonly keyBusinesses: BunkerKeyBusinesses;
    readonly contactsGovernance: BunkerContactsGovernance;

    readonly events: PageEventsProps;
    readonly financials: PageFinancialsProps;
    readonly sec: PageSecProps;
    readonly pressReleases: PagePressReleasesProps;
    readonly main: PageMainProps;
    readonly structure: BunkerStructure;
    readonly sustainability: BunkerSustainability;
    readonly bonds: BunkerBonds;
    readonly latestPresentation: BunkerLatestPresentation;
    readonly webcast: BunkerWebcast;
    readonly principles: PagePrinciplesProps;
    readonly directors: PageDirectorsProps;

    sectionTitle: string;
    baseUrl: string;
    nonce: string;
    faviconLang: 'en';
    yandexServiceId: string;
    pageName: PageName;
    langdetect: 'en';
    isTestingEnv: boolean;
    prefooter: PrefooterData;
    metrika: MetrikaData;
    menu: MenuItem[];
    stocksUrl: string;
    stocksTeaserUrl: string;
    emailFormUrl: string;
    askFormUrl: string;
    boardFormUrl: string;
    yandexFormEmbedPath: string;
}
