import cn from 'client/utils/cn';

import React, { Component } from 'react';

import Button from 'client/components/lego/button';

import { BunkerBonds, BunkerBondsDisclaimer, Doc } from 'client/bundles/types';
import Breadcrumbs from 'client/components/breadcrumbs';
import DocsList from 'client/components/docs-list';

import './index.css';

const b = cn('bonds');
const bPageTitle = cn('page-title');

interface Props {
    bonds?: BunkerBonds;
}

interface State {
    stage: 'DISCLAIMER' | 'CONTENT';
    disclaimerIndex: number;
}

class Bonds extends Component<Props, State> {

    constructor(props: Props) {
        super(props);

        this.state = {
            stage: 'DISCLAIMER',
            disclaimerIndex: 0
        };
    }

    onDisclaimerAcceptClick = () => {
        const disclaimersLength = this.props.bonds!.disclaimers.length;

        if (this.state.disclaimerIndex < (disclaimersLength - 1)) {
            this.setState({
                disclaimerIndex: this.state.disclaimerIndex + 1
            });
        } else if (this.state.disclaimerIndex === (disclaimersLength - 1)) {
            this.setState({
                stage: 'CONTENT'
            });
        }

        document.body.scrollIntoView();
    }

    onDisclaimerDeclineClick = () => {
        window.document.location.href = '/';
    }

    renderDisclaimer(disclaimers: BunkerBondsDisclaimer[]) {
        if (this.state.stage !== 'DISCLAIMER') {
            return null;
        }

        const disclaimer = disclaimers[this.state.disclaimerIndex];

        return (
            <div className={b('disclaimer')}>
                <div
                    className={b('disclaimer-text')}
                    dangerouslySetInnerHTML={{ __html: disclaimer.html }}
                    />
                <div className={b('disclaimer-buttons')}>
                    <Button
                        className={b('disclaimer-button')}
                        theme="action"
                        size="m"
                        onClick={this.onDisclaimerAcceptClick}
                        >
                        {disclaimer.buttons.accept.text}
                    </Button>

                    <Button
                        className={b('disclaimer-button')}
                        theme="normal"
                        size="m"
                        onClick={this.onDisclaimerDeclineClick}
                        >
                        {disclaimer.buttons.decline.text}
                    </Button>
                </div>
            </div>
        );
    }

    renderContent(docs: Doc[]) {
        if (this.state.stage !== 'CONTENT') {
            return null;
        }

        return (
            <div className={b('content')}>
                <DocsList docs={docs} />
            </div>
        );
    }

    render() {
        if (!this.props.bonds) {
            return null;
        }

        const { title, disclaimers, docs } = this.props.bonds;

        return (
            <div className={b()}>
                <Breadcrumbs link="/" text={'IR Home'} />

                <h1 className={bPageTitle({ type: 'border' })}>{title}</h1>

                {this.renderDisclaimer(disclaimers)}

                {this.renderContent(docs)}
            </div>
        );
    }
}

export default Bonds;
