import { Component, ReactNode } from 'react';

export const enum AutoscrollerStoragePrefix {
    SecFillings = 'sec-fillings',
    PressReleases = 'press-releases',
    Financials = 'financials',
    Events = 'events'
}

const SEPARATOR = '-/-';
const MOBILE_WIDTH = 480;
const SCROLL_GAP = 20;

const enum StorageKey {
    ScrollerPosition = 'scrollerPosition',
    LastClickedMenuItem = 'lastClickedMenuItem'
}

export type ClickEffect = (value: string) => void;
type ScrollProperty ='scrollLeft' | 'scrollTop';

interface OwnProps {
    currentMenuItem: HTMLElement | null;
    scroller: HTMLDivElement | null;
    prefix: AutoscrollerStoragePrefix;
    scrollProperty?: ScrollProperty;
    compare: (localStorageItem: string | null) => boolean;
    children: (onClickEffect: ClickEffect) => ReactNode;
}
interface DefaultProps {
    scrollProperty: ScrollProperty;
}

type Props = OwnProps & DefaultProps;

class Autoscroller extends Component<Props> {
    static defaultProps: DefaultProps = {
        scrollProperty: 'scrollLeft'
    };

    formatLocalStorageKey = (key: string) => {
        const { prefix } = this.props;

        return `${prefix}${SEPARATOR}${key}`;
    }

    setLocalStorage = (key: string, value: string) => {
        if (typeof localStorage === 'undefined') {
            return null;
        }

        localStorage.setItem(this.formatLocalStorageKey(key), value);
    }

    getLocalStorage = (key: string) => {
        if (typeof localStorage === 'undefined') {
            return null;
        }

        return localStorage.getItem(this.formatLocalStorageKey(key));
    }

    scrollToNeededPosition() {
        const scrollerPosition = this.getLocalStorage(StorageKey.ScrollerPosition);
        const lastClickedMenuItem = this.getLocalStorage(StorageKey.LastClickedMenuItem);
        const { scroller, currentMenuItem, scrollProperty } = this.props;

        if (!scroller) {
            return;
        }

        if (typeof scrollerPosition === 'string' && this.props.compare(lastClickedMenuItem)) {
            scroller[scrollProperty] = parseInt(scrollerPosition, 10);

            return;
        }

        if (!currentMenuItem) {
            return;
        }

        const { right, left } = currentMenuItem.getBoundingClientRect();

        if (window.innerWidth < MOBILE_WIDTH || right > window.innerWidth) {
            scroller[scrollProperty] = left - SCROLL_GAP;
        }
    }

    componentDidMount() {
        this.scrollToNeededPosition();
    }

    componentDidUpdate(prevProps: OwnProps) {
        if (prevProps.scroller !== this.props.scroller || prevProps.currentMenuItem !== this.props.currentMenuItem) {
            this.scrollToNeededPosition();
        }
    }

    setStorageOnClick = (value: string) => {
        const { scroller, scrollProperty } = this.props;

        this.setLocalStorage(StorageKey.ScrollerPosition, scroller?.[scrollProperty].toString() || '');
        this.setLocalStorage(StorageKey.LastClickedMenuItem, value);
    }

    render() {
        return this.props.children(this.setStorageOnClick);
    }

}

export default Autoscroller;
