import cn from 'client/utils/cn';

import React, { Component } from 'react';

import CopyButton from 'client/components/copy-button';

import './index.css';

const _ = require('lodash');

const b = cn('hint');
const enum HINT_DIRECTION_MOD {
    toLeft = 'to-left',
    toRight = 'to-right'
}

interface Props<T extends HTMLElement> {
    mixClassName: string;
    content?: string;
    parentNode: T | null;
    width: number;
    showCopyButton?: boolean;
}

interface State {
    direction: HINT_DIRECTION_MOD;
}

class Hint<T extends HTMLElement> extends Component<Props<T>, State> {
    constructor(props: Props<T>) {
        super(props);

        this.state = {
            direction: HINT_DIRECTION_MOD.toLeft
        };

        this.calcHintDirection = _.throttle(this.calcHintDirection.bind(this), 300);
    }

    componentDidMount() {
        this.calcHintDirection();

        window.addEventListener('resize', this.calcHintDirection);
    }

    componentDidUpdate(prevProps: Props<T>) {
        if (prevProps.parentNode !== this.props.parentNode) {
            this.calcHintDirection();
        }
    }

    componentWillUnmount() {
        window.removeEventListener('resize', this.calcHintDirection);
    }

    render() {
        const { mixClassName, children, content, showCopyButton } = this.props;

        if (!(children || content)) {
            return null;
        }

        const { direction } = this.state;

        return (
            <div className={b({ direction }, [mixClassName])}>
                {content ?
                    <span dangerouslySetInnerHTML={{ __html: content }} /> :
                    children
                }

                {content && showCopyButton && (
                    <div className={b('copy-button')}>
                        <CopyButton text={content} />
                    </div>
                )}
            </div>
        );
    }

    private tryToSetState(value: HINT_DIRECTION_MOD) {
        if (this.state.direction !== value) {
            this.setState({
                direction: value
            });
        }
    }

    private calcHintDirection() {
        const { parentNode: ref, width } = this.props;
        const nodeRect = ref?.getBoundingClientRect();

        if (!nodeRect) {
            return;
        }

        const windowWidth = document.documentElement.clientWidth;

        // Пробуем открыть всегда сначала вправо
        const predictedRightEdge = nodeRect.left + width;

        if (predictedRightEdge < windowWidth) {
            return this.tryToSetState(HINT_DIRECTION_MOD.toRight);
        }

        // Пробуем открыть влево, если не удалось открыть вправо
        const predictedLeftEdge = nodeRect.right - width;
        const MIN_LEFT_EDGE = 0;

        if (predictedLeftEdge > MIN_LEFT_EDGE) {
            return this.tryToSetState(HINT_DIRECTION_MOD.toLeft);
        }

        // Если никуда нельзя открыть, открываем вправо
        if (this.state.direction !== HINT_DIRECTION_MOD.toRight) {
            this.tryToSetState(HINT_DIRECTION_MOD.toRight);
        }

    }
}

export default Hint;
