import { BundleData } from 'client/bundles/types';
import cn from 'client/utils/cn';

import React, { Component } from 'react';

import Hint from 'client/components/hint';

import './index.css';

const b = cn('structure-node');
const TOP_LEVEL_TYPE = 'top-level-segment';

type Structure = BundleData['keyBusinesses']['structure'];
type TopLevelSegment = Structure[number];
type Subnode = TopLevelSegment['subnodes'][number];
type NodeType = Subnode['subnodes-type'] | typeof TOP_LEVEL_TYPE;

interface OwnProps {
    node: Subnode;
    type?: NodeType;
}

interface DefaultProps {
    type: NodeType;
}

type Props = OwnProps & DefaultProps;

class StructureNode extends Component<Props> {
    static defaultProps: DefaultProps = {
        type: TOP_LEVEL_TYPE
    };

    private nodeRef = React.createRef<HTMLDivElement>();
    private hintWidth: number;

    constructor(props: Props) {
        super(props);

        this.calcHintWidth = this.calcHintWidth.bind(this);
    }

    render() {
        const { node, type } = this.props;

        const { title, subnodes, 'subnodes-type': subnodesType, content } = node;
        const hasSubnodes = subnodes.length !== 0;
        const hasParent = type !== TOP_LEVEL_TYPE;

        return (
            <div className={b({ type, 'has-subnodes': hasSubnodes, 'has-parent': hasParent, 'subnodes-type': subnodesType })} ref={this.nodeRef}>
                {this.renderTitle(title, content)}
                <div className={b('subnodes')}>
                    {subnodes.map(
                        subnode => <StructureNode key={subnode.title} node={subnode} type={subnodesType} />
                    )}
                </div>
            </div>
        );
    }

    componentDidMount() {
        this.calcHintWidth();
        window.addEventListener('resize', this.calcHintWidth);
    }

    componentWillUnmount() {
        window.removeEventListener('resize', this.calcHintWidth);
    }

    private calcHintWidth() {
        const prevHintWidth = this.hintWidth;

        const windowWidth = Math.min(window.outerWidth, window.innerWidth);
        const maxHintWidth = windowWidth - 40;
        const settledHintWidth = windowWidth < 1366 ? 248 : 352;

        this.hintWidth = Math.min(maxHintWidth, settledHintWidth);

        if (prevHintWidth !== this.hintWidth) {
            this.forceUpdate();
        }
    }

    private renderTitle(title: string, hintContent?: string) {
        return (
            <div className={b('title')}>
                <div className={b('title-text', { 'has-hovered-content': Boolean(hintContent) })}>
                    <span dangerouslySetInnerHTML={{ __html: title }} />
                    <Hint
                        mixClassName={b('hint')}
                        content={hintContent}
                        parentNode={this.nodeRef.current}
                        width={this.hintWidth}
                        showCopyButton
                        />
                </div>
            </div>
        );
    }
}

export default StructureNode;
